package eu.dnetlib.data.collector.plugins.sftp;

import com.jcraft.jsch.*;
import eu.dnetlib.data.collector.rmi.CollectorServiceRuntimeException;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.*;

/**
 * Created by andrea on 11/01/16.
 */
public class SftpIterator implements Iterator<String> {
    private static final Log log = LogFactory.getLog(SftpIterator.class);

    private static final int MAX_RETRIES = 5;
    private static final int DEFAULT_TIMEOUT = 30000;
    private static final long BACKOFF_MILLIS = 10000;

    private String baseUrl;
    private String sftpURIScheme;
    private String sftpServerAddress;
    private String remoteSftpBasePath;
    private String username;
    private String password;
    private boolean isRecursive;
    private Set<String> extensionsSet;

    private Session sftpSession;
    private ChannelSftp sftpChannel;

    private Queue<String> queue;

    public SftpIterator(String baseUrl, String username, String password, boolean isRecursive, Set<String> extensionsSet) {
        this.baseUrl = baseUrl;
        this.username = username;
        this.password = password;
        this.isRecursive = isRecursive;
        this.extensionsSet = extensionsSet;
        try {
            URI sftpServer = new URI(baseUrl);
            this.sftpURIScheme = sftpServer.getScheme();
            this.sftpServerAddress = sftpServer.getHost();
            this.remoteSftpBasePath = sftpServer.getPath();
        } catch (URISyntaxException e) {
            throw new CollectorServiceRuntimeException("Bad syntax in the URL " + baseUrl);
        }

        connectToSftpServer();
        initializeQueue();
    }

    private void connectToSftpServer() {
        JSch jsch = new JSch();

        try {
            JSch.setConfig("StrictHostKeyChecking", "no");
            sftpSession = jsch.getSession(username, sftpServerAddress);
            sftpSession.setPassword(password);
            sftpSession.connect();

            Channel channel = sftpSession.openChannel(sftpURIScheme);
            channel.connect();
            sftpChannel = (ChannelSftp) channel;
            sftpChannel.cd(sftpChannel.pwd() + remoteSftpBasePath);
            log.info("Connected to SFTP server " + sftpServerAddress);
        } catch (JSchException e) {
            throw new CollectorServiceRuntimeException("Unable to connect to remote SFTP server.", e);
        } catch (SftpException e) {
            throw new CollectorServiceRuntimeException("Unable to access the base remote path on the SFTP server.", e);
        }
    }

    private void disconnectFromSftpServer() {
        sftpChannel.exit();
        sftpSession.disconnect();
    }

    private void initializeQueue() {
        queue = new LinkedList<String>();
        log.info(String.format("SFTP collector plugin collecting from %s with recursion = %s", remoteSftpBasePath, isRecursive));
        listDirectoryRecursive(".", "");
    }

    private void listDirectoryRecursive(final String parentDir, final String currentDir) {
        String dirToList = parentDir;
        if (!currentDir.equals("")) {
            dirToList += "/" + currentDir;
        }
        try {
            Vector<ChannelSftp.LsEntry> ls = sftpChannel.ls(dirToList);
            for (ChannelSftp.LsEntry entry : ls) {
                String currentFileName = entry.getFilename();
                if (currentFileName.equals(".") || currentFileName.equals("..")) {
                    // skip parent directory and directory itself
                    continue;
                }

                SftpATTRS attrs = entry.getAttrs();
                if (attrs.isDir()) {
                    if (isRecursive) {
                        listDirectoryRecursive(dirToList, currentFileName);
                    }
                } else {
                    // test the file for extensions compliance and, just in case, add it to the list.
                    for (String ext : extensionsSet) {
                        if (currentFileName.endsWith(ext)) {
                            queue.add(dirToList + "/" + currentFileName);
                        }
                    }
                }
            }
        } catch (SftpException e) {
            throw new CollectorServiceRuntimeException("Cannot list the sftp remote directory", e);

        }
    }

    @Override
    public boolean hasNext() {
        if (queue.isEmpty()) {
            disconnectFromSftpServer();
            return false;
        } else {
            return true;
        }
    }

    @Override
    public String next() {
        String nextRemotePath = queue.remove();
        int nRepeat = 0;

        while (nRepeat < MAX_RETRIES) {
            try {
                OutputStream baos = new ByteArrayOutputStream();
                /*if (!sftpChannel.isConnected() || !sftpSession.isConnected()) {
                    connectToSftpServer();
                }*/
                sftpChannel.get(nextRemotePath, baos);

                log.debug(String.format("Collected file from SFTP: %s%s", sftpServerAddress, nextRemotePath));
                return baos.toString();
            } catch (SftpException e) {
                nRepeat++;
                log.warn(String.format("An error occurred [%s] for %s%s, retrying.. [retried %s time(s)]", e.getMessage(), sftpServerAddress, nextRemotePath, nRepeat));
                // disconnectFromSftpServer();
                try {
                    Thread.sleep(BACKOFF_MILLIS);
                } catch (InterruptedException e1) {
                    log.error(e1);
                }
            }
        }
        throw new CollectorServiceRuntimeException(String.format("Impossible to retrieve FTP file %s after %s retries. Aborting FTP collection.", nextRemotePath, nRepeat));
    }

    @Override
    public void remove() {
        throw new UnsupportedOperationException();
    }
}
