package eu.dnetlib.functionality.modular.ui.repositories;

import java.io.StringReader;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import net.sf.ehcache.Cache;
import net.sf.ehcache.Element;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.common.collect.Maps;
import com.google.gson.Gson;

import eu.dnetlib.data.collector.rmi.CollectorService;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.ProtocolParameterValue;
import eu.dnetlib.enabling.datasources.rmi.BrowseTerm;
import eu.dnetlib.enabling.datasources.rmi.DatasourceDesc;
import eu.dnetlib.enabling.datasources.rmi.DatasourceManagerService;
import eu.dnetlib.enabling.datasources.rmi.DatasourceManagerServiceException;
import eu.dnetlib.enabling.datasources.rmi.IfaceDesc;
import eu.dnetlib.enabling.datasources.rmi.RepositoryMapEntry;
import eu.dnetlib.enabling.datasources.rmi.SearchInterfacesEntry;
import eu.dnetlib.enabling.datasources.rmi.SimpleDatasourceDesc;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.functionality.modular.ui.repositories.objects.RepoInterfaceEntry;
import eu.dnetlib.functionality.modular.ui.workflows.objects.sections.WorkflowSectionGrouper;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.msro.workflows.sarasvati.loader.WorkflowExecutor;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

@Controller
public class RepoInternalController {

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private WorkflowSectionGrouper workflowSectionGrouper;

	@Resource
	private WorkflowExecutor workflowExecutor;

	@Resource
	private RepoUIUtils repoUIUtils;

	@Resource(name = "repoUIJsonCache")
	private Cache repoUIJsonCache;

	private static final Log log = LogFactory.getLog(RepoInternalController.class);

	@RequestMapping(value = "/ui/browseRepoField.do")
	public @ResponseBody
	List<BrowseTerm> browseRepoField(@RequestParam(value = "field", required = true) final String field) throws Exception {
		return serviceLocator.getService(DatasourceManagerService.class).browseField(field);
	}

	@SuppressWarnings("unchecked")
	@RequestMapping(value = "/ui/listApis.do")
	public @ResponseBody
	List<SearchInterfacesEntry> listApis(
			@RequestParam(value = "param", required = true) final String param,
			@RequestParam(value = "value", required = true) final String value,
			@RequestParam(value = "refresh", required = false) final String refresh) throws Exception {

		final String cacheKey = "list@@@" + param + "@@@" + value;

		final Element elem = repoUIJsonCache.get(cacheKey);

		if (elem != null && refresh == null) {
			return (List<SearchInterfacesEntry>) elem.getObjectValue();
		} else {
			log.info("Refreshing " + cacheKey + " cache...");
			final List<SearchInterfacesEntry> list = serviceLocator.getService(DatasourceManagerService.class).searchInterface(param, value);
			repoUIJsonCache.put(new Element(cacheKey, list));
			return list;
		}
	}

	@RequestMapping(value = "/ui/listRepositories.map")
	public @ResponseBody
	List<RepositoryMapEntry> listRepositories_asMap() throws Exception {
		return serviceLocator.getService(DatasourceManagerService.class).getRepositoryMap();
	}

	@RequestMapping(value = "/ui/listRepositories.json")
	public @ResponseBody
	List<SimpleDatasourceDesc> listRepositories(@RequestParam(value = "type", required = true) final String type) throws Exception {
		return serviceLocator.getService(DatasourceManagerService.class).simpleListDatasourcesByType(type);
	}

	@RequestMapping(value = "/ui/validateRepo.do")
	public @ResponseBody
	String listRepositories(@RequestParam(value = "id", required = true) final String id,
			@RequestParam(value = "b", required = true) final boolean b) throws Exception {

		final String query = "count(/*[.//RESOURCE_TYPE/@value='MetaWorkflowDSResourceType' and .//DATAPROVIDER/@id='" + id + "'])";
		if (!b && Integer.parseInt(serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(query)) > 0) { throw new Exception("Repo " + id
				+ " can be invalidated: it is related to some metawfs"); }

		final String newId = b ?
				serviceLocator.getService(ISRegistryService.class).validateProfile(id) :
				serviceLocator.getService(ISRegistryService.class).invalidateProfile(id);

		repoUIJsonCache.removeAll();

		return newId;
	}

	@RequestMapping(value = "/ui/getRepoDetails.do")
	public void getRepoDetails(final HttpServletResponse response, @RequestParam(value = "id", required = true) final String id) throws Exception {
		String profile;
		try {
			profile = serviceLocator.getService(ISLookUpService.class).getResourceProfile(id);
		} catch (ISLookUpDocumentNotFoundException e) {
			profile = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
					"collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType')/*[.//DATASOURCE_ORIGINAL_ID='" + id + "']");
		}

		final ApplyXslt xslt = new ApplyXslt(IOUtils.toString(getClass().getResourceAsStream(
				"/eu/dnetlib/functionality/modular/ui/repositories/xslt/repoDetails.xslt")));

		IOUtils.copy(new StringReader(xslt.evaluate(profile)), response.getOutputStream());
	}

	@RequestMapping("/ui/repoMetaWf.new")
	public @ResponseBody
	String newDataProviderWorkflow(@RequestParam(value = "id", required = true) final String repoId,
			@RequestParam(value = "name", required = true) final String repoName,
			@RequestParam(value = "iface", required = true) final String ifaceId,
			@RequestParam(value = "wf", required = true) final String wfId) throws Exception {

		final Map<String, Object> params = Maps.newHashMap();
		params.put(WorkflowsConstants.DATAPROVIDER_ID, repoId);
		params.put(WorkflowsConstants.DATAPROVIDER_NAME, repoName);
		params.put(WorkflowsConstants.DATAPROVIDER_INTERFACE, ifaceId);

		return workflowExecutor.startProcess(wfId, params);
	}

	@RequestMapping("/ui/repoMetaWf.destroy")
	public @ResponseBody
	String destroyDataProviderWorkflow(@RequestParam(value = "destroyWf", required = true) final String destroyWfId)
			throws Exception {
		return workflowExecutor.startProcess(destroyWfId, null);
	}

	@RequestMapping("/ui/repoApi.get")
	public @ResponseBody
	RepoInterfaceEntry getRepoApi(@RequestParam(value = "repoId", required = true) final String repoId,
			@RequestParam(value = "ifaceId", required = true) final String ifaceId) throws Exception {
		return repoUIUtils.getApi(repoId, ifaceId);
	}

	@SuppressWarnings("unchecked")
	@RequestMapping("/ui/repoApi.update")
	public @ResponseBody
	boolean updateRepoApi(
			@RequestParam(value = "id", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceId,
			@RequestParam(value = "accessParams", required = true) final String accessParamsJson) throws Exception {
		if (!StringUtils.isEmpty(accessParamsJson)) {
			serviceLocator.getService(DatasourceManagerService.class).bulkUpdateApiAccessParams(repoId, ifaceId,
					new Gson().fromJson(accessParamsJson, Map.class));
		}
		return true;
	}

	@RequestMapping("/ui/repoApiCompliance.update")
	public @ResponseBody
	boolean updateRepoApiCompliance(@RequestParam(value = "id", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceId,
			@RequestParam(value = "compliance", required = true) final String compliance) throws Exception {

		log.debug("SET COMPLIANCE TO " + compliance);

		serviceLocator.getService(DatasourceManagerService.class).overrideCompliance(repoId, ifaceId, compliance);

		repoUIJsonCache.removeAll();

		return true;
	}

	@RequestMapping("/ui/repoApiCompliance.reset")
	public @ResponseBody
	boolean resetRepoApiCompliance(@RequestParam(value = "id", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceId) throws Exception {

		log.debug("RESET COMPLIANCE");

		serviceLocator.getService(DatasourceManagerService.class).overrideCompliance(repoId, ifaceId, null);

		repoUIJsonCache.removeAll();

		return true;
	}

	@RequestMapping("/ui/repos/repoApi.html")
	public void resetRepoApiCompliance(final ModelMap map) throws Exception {}

	@RequestMapping("/ui/repoApi.new")
	public @ResponseBody
	boolean addRepoApi(@RequestParam(value = "repoId", required = true) final String repoId,
			@RequestParam(value = "iface", required = true) final String ifaceJson) throws DatasourceManagerServiceException {
		final IfaceDesc iface = new Gson().fromJson(ifaceJson, IfaceDesc.class);

		log.info("Adding interface " + iface.getId() + " to repository " + repoId);

		final DatasourceManagerService dsManager = serviceLocator.getService(DatasourceManagerService.class);
		return dsManager.addInterface(repoId, iface);
	}

	@RequestMapping("/ui/repo.new")
	public @ResponseBody
	boolean addRepoApi(@RequestParam(value = "repo", required = true) final String repoJson) throws DatasourceManagerServiceException {
		final DatasourceDesc ds = new Gson().fromJson(repoJson, DatasourceDesc.class);
		ds.setDateOfCollection(new Date());

		if (StringUtils.isBlank(ds.getEnglishName())) {
			ds.setEnglishName(ds.getOfficialName());
		}

		log.info("Adding datasource " + ds.getId() + " - name " + ds.getOfficialName());

		final DatasourceManagerService dsManager = serviceLocator.getService(DatasourceManagerService.class);

		return dsManager.addDatasource(ds);
	}

	@RequestMapping("/ui/listValidValuesForParam.do")
	public @ResponseBody
	List<ProtocolParameterValue> listValidValuesForParam(
			@RequestParam(value = "protocol", required = true) final String protocol,
			@RequestParam(value = "param", required = true) final String param,
			@RequestParam(value = "baseUrl", required = true) final String baseUrl) throws CollectorServiceException {

		return serviceLocator.getService(CollectorService.class).listValidValuesForParam(protocol, baseUrl, param, null);
	}
}
