package eu.dnetlib.data.oai.store.mongo;

import java.util.Iterator;

import com.google.common.collect.Lists;
import com.mongodb.DBObject;
import com.mongodb.client.MongoCursor;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.data.oai.store.Cursor;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class DNetOAIMongoCursor implements Cursor {

	/**
	 * Underlying mongo cursor.
	 */
	private MongoCursor<DBObject> dbCursor;
	private int size = 0;
	/**
	 * Function to apply to records before delivering.
	 */
	private UnaryFunction<String, String> function;

	/**
	 * true if the RecordInfo returned by this Cursor must include the record body, false otherwise.
	 */
	private boolean bodyIncluded;

	private RecordInfoGenerator recordInfoGenerator;
	private MetadataExtractor metadataExtractor;
	private ProvenanceExtractor provenanceExtractor;

	public DNetOAIMongoCursor() {
		super();
	}

	public DNetOAIMongoCursor(final MongoCursor<DBObject> dbCursor, final boolean bodyIncluded, final RecordInfoGenerator recordInfoGenerator,
			final MetadataExtractor metadataExtractor) {
		this(dbCursor, null, bodyIncluded, recordInfoGenerator, metadataExtractor);
	}

	public DNetOAIMongoCursor(final MongoCursor<DBObject> dbCursor, final UnaryFunction<String, String> function, final boolean bodyIncluded,
			final RecordInfoGenerator recordInfoGenerator, final MetadataExtractor metadataExtractor) {
		super();
		this.dbCursor = dbCursor;
		this.size = 0;
		this.function = function;
		this.bodyIncluded = bodyIncluded;
		this.recordInfoGenerator = recordInfoGenerator;
		this.metadataExtractor = metadataExtractor;
	}

	/**
	 *
	 * {@inheritDoc}
	 */
	@Override
	public int count() {
		//I can do it because MongoCursor are always created from queries with "limit", so I do not expect the creation of the list to explode
		//to not exagerate, I'll get the size only if the current size is 0
		if (size == 0)
			size = Lists.newArrayList(dbCursor).size();
		return size;
	}

	/**
	 *
	 * {@inheritDoc}
	 *
	 * @see java.lang.Iterable#iterator()
	 */
	@Override
	public Iterator<RecordInfo> iterator() {

		return new Iterator<RecordInfo>() {

			@Override
			public boolean hasNext() {
				return dbCursor.hasNext();
			}

			@Override
			public RecordInfo next() {
				DBObject res = dbCursor.next();
				RecordInfo info = recordInfoGenerator.transformDBObject(res, bodyIncluded);
				if ((function != null) && bodyIncluded && (info != null)) {
					info.setMetadata(function.evaluate(info.getMetadata()));
				}
				return info;
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}

		};
	}

	public UnaryFunction<String, String> getFunction() {
		return function;
	}

	public void setFunction(final UnaryFunction<String, String> function) {
		this.function = function;
	}

	public MongoCursor<DBObject> getDbCursor() {
		return dbCursor;
	}

	public void setDbCursor(final MongoCursor<DBObject> dbCursor) {
		this.dbCursor = dbCursor;
	}

	@Override
	public boolean isBodyIncluded() {
		return this.bodyIncluded;
	}

	@Override
	public void setBodyIncluded(final boolean bodyIncluded) {
		this.bodyIncluded = bodyIncluded;
	}

	public RecordInfoGenerator getRecordInfoGenerator() {
		return recordInfoGenerator;
	}

	public void setRecordInfoGenerator(final RecordInfoGenerator recordInfoGenerator) {
		this.recordInfoGenerator = recordInfoGenerator;
	}

	public MetadataExtractor getMetadataExtractor() {
		return metadataExtractor;
	}

	public void setMetadataExtractor(final MetadataExtractor metadataExtractor) {
		this.metadataExtractor = metadataExtractor;
	}

	public ProvenanceExtractor getProvenanceExtractor() {
		return provenanceExtractor;
	}

	public void setProvenanceExtractor(final ProvenanceExtractor provenanceExtractor) {
		this.provenanceExtractor = provenanceExtractor;
	}

}
