package eu.dnetlib.data.oai.store.sync;

import java.util.concurrent.Callable;
import javax.annotation.Resource;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationExistReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.oai.store.PublisherStoreDAO;
import eu.dnetlib.data.oai.store.mongo.MongoPublisherStore;
import eu.dnetlib.data.oai.store.mongo.MongoPublisherStoreDAO;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class OAIStoreSynchronizer {

	private static final Log log = LogFactory.getLog(OAIStoreSynchronizer.class); // NOPMD by marko on 11/24/08 5:02 PM
	/**
	 * OAI Publisher configuration.
	 */
	@Resource
	private OAIConfigurationExistReader configuration;

	@Autowired
	private MongoPublisherStoreDAO mongoPublisherStoreDAO;

	public void synchronize(final Iterable<String> records,
			final MDFInfo sourceMetadataFormat,
			final String recordSource,
			final String dbName,
			final boolean alwaysNewRecord,
			final Callable<?> callback,
			final Callable<?> failCallback) {
		try {
			log.fatal("Synchronizing content for source metadata format " + sourceMetadataFormat);
			MongoPublisherStore store = this.getStore(sourceMetadataFormat, dbName, alwaysNewRecord);
			int count = store.feed(records, recordSource);
			log.info("Content synchronized: store " + sourceMetadataFormat + " fed with " + count + " records");
			executeCallback(callback);
		} catch (Exception e) {
			log.error(e);
			executeCallback(failCallback);
		}
	}

	/**
	 * Gets the OAI store for the given source metadata format. If the store does not exists, then a new one is created.
	 * 
	 * @param sourceMetadataFormat
	 *            MDFInfo about the metadata format of the store to get
	 * @return a MongoPublisherStore instance
	 */
	private MongoPublisherStore getStore(final MDFInfo sourceMetadataFormat, final String dbName, final boolean alwaysNewRecord) {
		this.mongoPublisherStoreDAO.setAlwaysNewRecord(alwaysNewRecord);
		MongoPublisherStore store = this.mongoPublisherStoreDAO.getStore(sourceMetadataFormat.getSourceFormatName(),
				sourceMetadataFormat.getSourceFormatInterpretation(), sourceMetadataFormat.getSourceFormatLayout(), dbName);
		if (store == null) {
			log.debug("Creating store for metadata format: \n" + sourceMetadataFormat + " in db: " + dbName);
			try {
				store = this.mongoPublisherStoreDAO.createStore(sourceMetadataFormat.getSourceFormatName(), sourceMetadataFormat.getSourceFormatInterpretation(),
						sourceMetadataFormat.getSourceFormatLayout(), dbName);
				log.debug("Created store with id: " + store.getId());
			} catch (OaiPublisherException e) {
				throw new OaiPublisherRuntimeException(e);
			}
		}
		return store;
	}

	protected void executeCallback(final Callable<?> callback) {
		if (callback != null) {
			try {
				callback.call();
			} catch (Exception e) {
				log.error("Error executing callback", e);
			}
		}
	}

	public OAIConfigurationExistReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final OAIConfigurationExistReader configuration) {
		this.configuration = configuration;
	}

}
