package eu.dnetlib.data.oai.store.mongo;

import java.util.List;
import javax.annotation.Resource;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.mongodb.DBObject;
import com.mongodb.MongoClient;
import com.mongodb.WriteConcern;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.oai.store.RecordChangeDetector;
import eu.dnetlib.data.oai.store.sets.MongoSetCollection;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;

/**
 * Created by alessia on 19/07/16.
 */
public class MongoPublisherCacheHelper {

	private static final Log log = LogFactory.getLog(MongoPublisherCacheHelper.class);

	@Autowired
	private MongoClient publisherMongoClient;
	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configuration;
	@Resource
	private MetadataExtractor metadataExtractor;
	@Resource
	private RecordInfoGenerator recordInfoGenerator;
	@Resource
	private RecordChangeDetector recordChangeDetector;


	public  MongoDatabase getDB(final String dbName) {
		return this.publisherMongoClient.getDatabase(dbName).withWriteConcern(WriteConcern.JOURNALED);
	}

	@Cacheable(value="oaistores", key="#dbname")
	public List<MongoPublisherStore> listPublisherStores(final String dbName, final String metadataCollectionName, final boolean alwaysNewRecord, final MongoSetCollection mongoSetCollection) {
		log.info("Not using cache for listPublisherStores on "+dbName);
		final MongoDatabase db = getDB(dbName);
		final FindIterable<DBObject> stores = db.getCollection(metadataCollectionName, DBObject.class).find();
		return Lists.newArrayList(
				Iterables.transform(stores, new Function<DBObject, MongoPublisherStore>() {
					@Override
					public MongoPublisherStore apply(final DBObject storeInfo) {
						return createFromDBObject(storeInfo, db, alwaysNewRecord, mongoSetCollection);
					}
				})
		);
	}

	@Cacheable(value="oaistoresById", key="#storeId + #dbName", unless="#result == null")
	public MongoPublisherStore getStoreById(final String storeId, final String dbName, final String metadataCollectionName, final boolean alwaysNewRecord, final MongoSetCollection mongoSetCollection) {
		log.info(String.format("Not using cache for getStoreById: %s", storeId));
		DBObject storeInfo = getDB(dbName).getCollection(metadataCollectionName, DBObject.class).find(Filters.eq("id", storeId)).first();
		log.info("Got DBObject from mongo "+dbName+"."+metadataCollectionName+", id "+storeId+" is : "+storeInfo );
		return this.createFromDBObject(storeInfo, getDB(dbName), alwaysNewRecord, mongoSetCollection);
	}

	@CacheEvict(value="oaistoresById", key = "#storeId + #dbName")
	public void deleteFromCache(String storeId, String dbName){
		log.info("Evicting "+storeId+" for db "+dbName+ " from the cache");
	}


	private MongoPublisherStore createFromDBObject(final DBObject storeInfo, final MongoDatabase db, final boolean alwaysNewRecord, final MongoSetCollection mongoSetCollection) {
		if (storeInfo == null){
			log.error("cannot create MongoPublisherStore from null DBObject");
			return null;
		}
		log.debug("Creating MongoPublisherStore from DBObject "+storeInfo.toString());
		String storeId = (String) storeInfo.get("id");
		String mdFormat = (String) storeInfo.get("metadataFormat");
		String mdInterpreation = (String) storeInfo.get("interpretation");
		String mdLayout = (String) storeInfo.get("layout");
		MongoPublisherStore store = new MongoPublisherStore(storeId, mdFormat, mdInterpreation, mdLayout, db.getCollection(storeId, DBObject.class),
				this.configuration.getFields(mdFormat, mdInterpreation, mdLayout), recordInfoGenerator, this.configuration.getIdScheme(),
				this.configuration.getIdNamespace(), this.metadataExtractor, this.recordChangeDetector, alwaysNewRecord, db);
		store.setMongoSetCollection(mongoSetCollection);
		return store;
	}
}
