package eu.dnetlib.data.oai.store.mongo;

import java.util.List;
import javax.annotation.Resource;

import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.oai.store.PublisherStoreDAO;
import eu.dnetlib.data.oai.store.sets.MongoSetCollection;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

public class MongoPublisherStoreDAO implements PublisherStoreDAO<MongoPublisherStore, DNetOAIMongoCursor> {

	private static final Log log = LogFactory.getLog(MongoPublisherStoreDAO.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Autowired
	private MongoPublisherCacheHelper mongoCacheHelper;

	/** Name of the collection with information about the OAI stores. **/
	private String metadataCollection;

	@Resource
	private MongoSetCollection mongoSetCollection;

	private boolean alwaysNewRecord;

	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configuration;


	@Override
	public List<MongoPublisherStore> listPublisherStores(final String dbName) {
		return mongoCacheHelper.listPublisherStores(dbName, metadataCollection, alwaysNewRecord, mongoSetCollection);
	}

	@Override
	public MongoPublisherStore getStore(final String storeId, final String dbName) {
		return mongoCacheHelper.getStoreById(storeId, dbName, metadataCollection, alwaysNewRecord,  mongoSetCollection);
	}

	@Override
	public MongoPublisherStore getStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName) {
		return this.getStore(this.generateStoreId(mdfName, mdfInterpretation, mdfLayout), dbName);
	}

	@Override
	public MongoPublisherStore getStoreFor(final String targetMetadataPrefix, final String dbName) {
		MDFInfo info = this.configuration.getMetadataFormatInfo(targetMetadataPrefix);
		MongoPublisherStore store = this.getStore(info.getSourceFormatName(), info.getSourceFormatInterpretation(), info.getSourceFormatLayout(), dbName);
		return store;
	}

	@Override
	public MongoPublisherStore createStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName)
			throws OaiPublisherException {
		log.debug("CREATING OAI STORE");
		MongoDatabase db = this.mongoCacheHelper.getDB(dbName);
		DBObject store = createMetadataEntry(mdfName, mdfInterpretation, mdfLayout);
		MongoCollection<DBObject> metadata = db.getCollection(this.metadataCollection, DBObject.class);
		String id = (String) store.get("id");
		metadata.insertOne(store);
		log.debug("metadata insert done for store "+id);
		MongoPublisherStore theStore = this.mongoCacheHelper.getStoreById(id, dbName, this.metadataCollection, this.alwaysNewRecord, this.mongoSetCollection);
		return theStore;

	}

	@Override
	public boolean deleteStore(final String storeId, final String dbName) {
		log.debug("DELETING OAI STORE "+storeId+" -- db: "+dbName);
		this.mongoCacheHelper.deleteFromCache(storeId, dbName);
		MongoDatabase db = this.mongoCacheHelper.getDB(dbName);
		MongoCollection<DBObject> metadata = db.getCollection(this.metadataCollection, DBObject.class);
		final DBObject storeDeleted = metadata.findOneAndDelete(Filters.eq("id", storeId));
		if (storeDeleted == null) return false;
		else {
			db.getCollection(storeId).drop();
			// TODO: should drop entries related to mdPrefix served by the store we are deleting, not all of them.
			this.mongoSetCollection.dropOAISets(dbName);
			log.debug("Deleted oaistore " + storeId + ", db: " + dbName);
			return true;
		}
	}

	@Override
	public boolean deleteFromStore(final String storeId, final String dbName, final String set) {
		log.debug(String.format("DELETING OAI STORE %s BY SET %s from db %s", storeId, set, dbName));
		this.mongoCacheHelper.deleteFromCache(storeId,dbName);
		MongoDatabase db = this.mongoCacheHelper.getDB(dbName);
		MongoCollection<DBObject> metadata = db.getCollection(this.metadataCollection, DBObject.class);
		DBObject storeInfo = metadata.find(Filters.eq("id", storeId)).first();
		if (storeInfo == null) return false;
		else {
			db.getCollection(storeId).deleteOne(Filters.eq(OAIConfigurationReader.SET_FIELD, set));
			this.mongoSetCollection.dropSet(dbName, set);
			log.debug("Deleted set " + set + " from oaistore " + storeId + ", db: " + dbName);
			return true;
		}
	}

	@Override
	public boolean deleteFromStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName, final String set) {
		return this.deleteFromStore(this.generateStoreId(mdfName, mdfInterpretation, mdfLayout), dbName, set);
	}

	@Override
	public boolean deleteStore(final String mdfName, final String mdfInterpretation, final String mdfLayout, final String dbName) {
		return this.deleteStore(this.generateStoreId(mdfName, mdfInterpretation, mdfLayout), dbName);
	}

	public void ensureIndex(final MongoPublisherStore store) {
		if (store == null) throw new OaiPublisherRuntimeException("Can't ensure index on null store");
		Thread t = new Thread() {

			@Override
			public void run() {
				store.ensureIndices();
			}
		};
		t.start();
	}

	public void ensureIndex(final String dbName) {
		List<MongoPublisherStore> stores = this.listPublisherStores(dbName);
		for (final MongoPublisherStore s : stores) {
			s.ensureIndices();
		}
	}



	private DBObject createMetadataEntry(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		DBObject info = BasicDBObjectBuilder.start("id", generateStoreId(mdfName, mdfInterpretation, mdfLayout)).append("metadataFormat", mdfName)
				.append("interpretation", mdfInterpretation).append("layout", mdfLayout).get();
		log.debug("Created DBObject for OAIStore metadata entry: "+info.toString());
		return info;

	}

	private String generateStoreId(final String mdfName, final String mdfInterpretation, final String mdfLayout) {
		return mdfName + "-" + mdfLayout + "-" + mdfInterpretation;
	}

	public String getMetadataCollection() {
		return metadataCollection;
	}

	@Required
	public void setMetadataCollection(final String metadataCollection) {
		this.metadataCollection = metadataCollection;
	}

	public OAIConfigurationReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final OAIConfigurationReader configuration) {
		this.configuration = configuration;
	}

	public MongoSetCollection getMongoSetCollection() {
		return mongoSetCollection;
	}

	public void setMongoSetCollection(final MongoSetCollection mongoSetCollection) {
		this.mongoSetCollection = mongoSetCollection;
	}

	public boolean isAlwaysNewRecord() {
		return alwaysNewRecord;
	}

	public void setAlwaysNewRecord(final boolean alwaysNewRecord) {
		this.alwaysNewRecord = alwaysNewRecord;
	}

}
