package eu.dnetlib.data.transform.xml;

import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.Descriptor;

import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.OAIProvenance;
import eu.dnetlib.data.proto.FieldTypeProtos.OAIProvenance.OriginDescription;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.PersonPersonProtos.PersonPerson;
import eu.dnetlib.data.proto.PersonPersonProtos.PersonPerson.CoAuthorship;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.PersonProtos.Person.CoAuthor;
import eu.dnetlib.data.proto.PersonResultProtos.PersonResult;
import eu.dnetlib.data.proto.PersonResultProtos.PersonResult.Authorship;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.RelTypeProtos.SubRelType;
import eu.dnetlib.data.proto.ResultProjectProtos.ResultProject;
import eu.dnetlib.data.proto.ResultProjectProtos.ResultProject.Outcome;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.proto.ResultResultProtos.ResultResult;
import eu.dnetlib.data.proto.ResultResultProtos.ResultResult.PublicationDataset;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class DmfToHbaseXsltFunctions extends AbstractDNetOafXsltFunctions {

	private static final int MAX_COAUTHORS = 50;

	// dnet:oafPersonResultFromDMF($resultId, $oafPerson, position(), "sysimport:crosswalk:repository", "0.9")
	public static String oafPersonResult_Authorship_FromDMF(final String source,
			final String target,
			final int rank,
			final String relClass,
			final String provenanceAction,
			final String trust,
			final NodeList about) {
		try {
			final String eSource = OafRowKeyDecoder.decode(source).getKey();
			final String eTarget = OafRowKeyDecoder.decode(target).getKey();

			final Authorship.Builder auth = Authorship.newBuilder().setRanking("" + rank)
					.setRelMetadata(RelMetadata.newBuilder().setSemantics(getSimpleQualifier(relClass, "dnet:personroles")));

			final OafRel.Builder rel = getRel(eSource, eTarget, RelType.personResult, SubRelType.authorship, relClass, false).setPersonResult(
					PersonResult.newBuilder().setAuthorship(auth));

			return base64(getOaf(rel, getDataInfo(about, provenanceAction, trust, false, false)).toByteArray());
		} catch (final Throwable e) {
			System.err.println("source: " + source);
			System.err.println("target: " + target);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);
			System.err.println("rank: " + rank);
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	// dnet:oafPersonPersonFromMDStore($personId, $coauthorId)
	public static String oafPersonPerson_CoAuthorship_FromDMF(final String source,
			final String target,
			final String relClass,
			final String provenanceAction,
			final String trust,
			final NodeList about) {
		try {
			final String eSource = OafRowKeyDecoder.decode(source).getKey();
			final String eTarget = OafRowKeyDecoder.decode(target).getKey();

			final PersonPerson.Builder pp = PersonPerson.newBuilder();
			final CoAuthorship.Builder coauth = CoAuthorship.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(relClass, "dnet:personroles")));

			final OafRel.Builder rel = getRel(eSource, eTarget, RelType.personPerson, SubRelType.coauthorship, relClass, false).setPersonPerson(
					pp.setCoauthorship(coauth));

			return base64(getOaf(rel, getDataInfo(about, provenanceAction, trust, false, false)).toByteArray());
		} catch (final Throwable e) {
			System.err.println("source: " + source);
			System.err.println("target: " + target);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	// dnet:oafPersonFromDMF($personId, ., "sysimport:crosswalk:repository", "0.9")

	// collectedFromId, collectedFromName, originalId, dateOfCollection
	public static String oafPerson_FromDMF(final String personId,
			final String fullname,
			final NodeList authors,
			final String namespaceprefix,
			final String objIdentifier,
			final Map<String, Object> map,
			final String provenanceAction,
			final String trust,
			final NodeList about,
			final String collectedFromId,
			final String collectedFromName,
			final String originalId,
			final String dateOfCollection) {
		try {
			final String entityId = OafRowKeyDecoder.decode(personId).getKey();

			final Person.Builder person = Person.newBuilder();
			final Person.Metadata.Builder metadata = getMetadata(fullname);

			if (authors != null) {
				for (int i = 0; (i < authors.getLength()) && (i < MAX_COAUTHORS); i++) {
					final Node node = authors.item(i);

					final String name = StringUtils.trim(node.getTextContent());
					if (!name.equals(fullname)) {

						final CoAuthor.Builder coAuthor = CoAuthor.newBuilder();
						coAuthor.setId(oafPersonId("person", namespaceprefix, objIdentifier, name, map));
						coAuthor.setMetadata(getMetadata(name));

						person.addCoauthor(coAuthor);
					}
				}
			}

			// metadata.setNationality(getSimpleQualifier("UNKNOWN", "dnet:countries"));
			final List<StructuredProperty> pids = Lists.newArrayList(getStructuredProperty(originalId, "oai", "oai", "dnet:pid_types", "dnet:pid_types"));
			final OafEntity.Builder entity = getEntity(Type.person, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection, pids)
					.setPerson(person.setMetadata(metadata));

			final Oaf oaf = getOaf(entity, getDataInfo(about, provenanceAction, trust, false, false));
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			System.err.println("personId: " + personId);
			System.err.println("fullname: " + fullname);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);
			System.err.println("collectedFromId: " + collectedFromId);
			System.err.println("collectedFromName: " + collectedFromName);
			System.err.println("originalId: " + originalId);
			System.err.println("dateOfCollection: " + dateOfCollection);
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	private static Person.Metadata.Builder getMetadata(final String fullname) {
		final Person.Metadata.Builder metadata = Person.Metadata.newBuilder();

		metadata.setFullname(sf(fullname));

		final eu.dnetlib.pace.model.Person p = new eu.dnetlib.pace.model.Person(fullname, false);
		if (p.isAccurate()) {
			metadata.setFirstname(sf(p.getNormalisedFirstName()));
			metadata.clearSecondnames().addSecondnames(sf(p.getNormalisedSurname()));
			// metadata.setFullname(sf(p.getNormalisedFullname()));
		}
		return metadata;
	}

	// dnet:oafResultProjectFromDMF($resultId, $projectId, "sysimport:crosswalk:repository", "0.9")
	public static String oafResultProject_Outcome_FromDMF(final String source,
			final String target,
			final String relClass,
			final String provenanceAction,
			final String trust,
			final NodeList about) {
		try {
			final String eSource = OafRowKeyDecoder.decode(source).getKey();
			final String eTarget = OafRowKeyDecoder.decode(target).getKey();

			final Outcome.Builder outcome = Outcome.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(relClass, "dnet:result_project_relations")));

			final ResultProject.Builder rp = ResultProject.newBuilder().setOutcome(outcome);

			final OafRel.Builder rel = getRel(eSource, eTarget, RelType.resultProject, SubRelType.outcome, relClass, false).setResultProject(rp);

			return base64(getOaf(rel, getDataInfo(about, provenanceAction, trust, false, false)).toByteArray());
		} catch (final Throwable e) {
			System.err.println("source: " + source);
			System.err.println("target: " + target);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);

			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	// dnet:oafResultProjectFromDMF($resultId, $projectId, "sysimport:crosswalk:repository", "0.9")
	public static String oafResultProject_Outcome_FromDMF(final String source,
			final String target,
			final String relClass,
			final String provenanceAction,
			final String trust) {
		return oafResultProject_Outcome_FromDMF(source, target, relClass, provenanceAction, trust, null);
	}

	public static String oafResultResult_PublicationDataset_FromDMF(final String source,
			final String target,
			final String relClass,
			final String provenanceAction,
			final String trust) {
		return oafResultResult_PublicationDataset_FromDMF(source, target, relClass, provenanceAction, trust, null);
	}

	public static String oafResultResult_PublicationDataset_FromDMF(final String source,
			final String target,
			final String relClass,
			final String provenanceAction,
			final String trust,
			final NodeList about) {
		try {
			final String eSource = OafRowKeyDecoder.decode(source).getKey();
			final String eTarget = OafRowKeyDecoder.decode(target).getKey();

			final PublicationDataset.Builder pd = PublicationDataset.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(relClass, "dnet:result_result_relations")));

			final ResultResult.Builder rr = ResultResult.newBuilder().setPublicationDataset(pd);

			final OafRel.Builder rel = getRel(eSource, eTarget, RelType.resultResult, SubRelType.publicationDataset, relClass, false).setResultResult(rr);

			return base64(getOaf(rel, getDataInfo(about, provenanceAction, trust, false, false)).toByteArray());
		} catch (final Throwable e) {
			System.err.println("source: " + source);
			System.err.println("target: " + target);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);

			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	public static String oafResult_FromDMF(final String resultId,
			final String provenanceAction,
			final String trust,
			final NodeList about,
			final String hostedbyId,
			final String hostedbyName,
			final String collectedFromId,
			final String collectedFromName,
			final String originalId,
			final String dateOfCollection,
			final NodeList nodelist) {
		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();

			final Result.Builder result = Result.newBuilder();

			final ValueMap values = ValueMap.parseNodeList(nodelist);

			final Result.Metadata.Builder metadata = Result.Metadata.newBuilder();
			final Descriptor mDesc = Result.Metadata.getDescriptor();

			if (values.get("creator") != null) {
				for (final String fullname : Iterables.limit(values.get("creator").listValues(), 10)) {

					final Person.Metadata.Builder authorMetadata = Person.Metadata.newBuilder();

					authorMetadata.setFullname(sf(fullname));

					final eu.dnetlib.pace.model.Person p = new eu.dnetlib.pace.model.Person(fullname, false);
					if (p.isAccurate()) {
						authorMetadata.setFirstname(sf(p.getNormalisedFirstName()));
						authorMetadata.clearSecondnames().addSecondnames(sf(p.getNormalisedSurname()));
						authorMetadata.setFullname(sf(p.getNormalisedFullname()));
					}

					result.addAuthor(Person.newBuilder().setMetadata(authorMetadata));
				}
			}

			addStructuredProps(metadata, mDesc.findFieldByName("subject"), values.get("subject").listValues(), "keyword", "dnet:result_subject");
			addStructuredProps(metadata, mDesc.findFieldByName("title"), values.get("title").listValues(), "main title", "dnet:dataCite_title");

			for (final String fieldname : Lists.newArrayList("description", "source")) {
				if (values.get(fieldname) != null) {
					for (final String s : values.get(fieldname).listValues()) {
						addField(metadata, mDesc.findFieldByName(fieldname), s);
					}
				}
			}

			addField(metadata, mDesc.findFieldByName("language"), setQualifier(getDefaultQualifier("dnet:languages"), values.get("language").listValues()));
			addField(metadata, mDesc.findFieldByName("dateofacceptance"), values.get("dateaccepted").listValues());
			addField(metadata, mDesc.findFieldByName("publisher"), values.get("publisher").listValues());
			addField(metadata, mDesc.findFieldByName("embargoenddate"), values.get("embargoenddate").listValues());
			addField(metadata, mDesc.findFieldByName("storagedate"), values.get("storagedate").listValues());

			addField(metadata, mDesc.findFieldByName("resulttype"), getSimpleQualifier("publication", "dnet:result_typologies"));

			addField(metadata, mDesc.findFieldByName("fulltext"), values.get("fulltext").listValues());
			addField(metadata, mDesc.findFieldByName("format"), values.get("format").listValues());

			// addField(metadata, Result.Metadata.getDescriptor().findFieldByName("provenanceaction"),
			// getSimpleQualifier("sysimport:crosswalk:repository", "dnet:provenanceActions").build());

			if (values.get("concept") != null) {
				for (final Element e : values.get("concept")) {
					final String id = e.getAttributes().get("id");
					if (StringUtils.isBlank(id)) throw new IllegalArgumentException("Context id cannot be blank");
					metadata.addContext(Context.newBuilder().setId(id));
				}
			}

			if (values.get("journal") != null) {
				for (final Element e : values.get("journal")) {

					final Journal.Builder journal = Journal.newBuilder();
					if (e.getText() != null) {
						journal.setName(e.getText());
					}

					final Map<String, String> attr = e.getAttributes();
					if (attr != null) {
						if (attr.get("issn") != null) {
							journal.setIssnPrinted(attr.get("issn"));
						}
						if (attr.get("eissn") != null) {
							journal.setIssnOnline(attr.get("eissn"));
						}
						if (attr.get("lissn") != null) {
							journal.setIssnLinking(attr.get("lissn"));
						}
					}
					metadata.setJournal(journal.build());
				}
			}

			final Instance.Builder instance = Instance.newBuilder().setHostedby(getKV(hostedbyId, hostedbyName));

			addField(instance, Instance.getDescriptor().findFieldByName("licence"),
					setQualifier(getDefaultQualifier("dnet:access_modes"), values.get("accessrights").listValues()));
			addField(instance, Instance.getDescriptor().findFieldByName("instancetype"),
					setQualifier(getDefaultQualifier("dnet:publication_resource"), values.get("cobjcategory").listValues()));

			if (values.get("identifier") != null) {
				addField(instance, Instance.getDescriptor().findFieldByName("url"),
						Lists.newArrayList(Iterables.filter(values.get("identifier").listValues(), urlFilter)));
			}

			result.addInstance(instance);

			final List<Element> extrefs = values.get("reference");
			if (!extrefs.isEmpty()) {
				final Descriptor extDesc = ExternalReference.getDescriptor();
				for (final Element element : extrefs) {
					final ExternalReference.Builder extref = ExternalReference.newBuilder();
					addField(extref, extDesc.findFieldByName("url"), element.getText());
					addField(extref, extDesc.findFieldByName("sitename"), element.getAttributes().get("source"));
					addField(extref, extDesc.findFieldByName("refidentifier"), element.getAttributes().get("identifier"));
					addField(extref, extDesc.findFieldByName("label"), element.getAttributes().get("title"));
					addField(extref, extDesc.findFieldByName("query"), element.getAttributes().get("query"));
					addField(extref, extDesc.findFieldByName("qualifier"),
							setQualifier(getDefaultQualifier("dnet:externalReference_typologies"), Lists.newArrayList(element.getAttributes().get("type")))
							.build());

					result.addExternalReference(extref);
				}
			}

			final List<StructuredProperty> pids = Lists.newArrayList();
			pids.addAll(parsePids(nodelist));
			pids.add(getStructuredProperty(originalId, "oai", getClassName("oai"), "dnet:pid_types", "dnet:pid_types"));

			final OafEntity.Builder entity = getEntity(Type.result, entityId, getKV(collectedFromId, collectedFromName), originalId, dateOfCollection, pids)
					.setResult(result.setMetadata(metadata));

			entity.setOaiprovenance(getOAIProvenance(about));

			final Oaf oaf = getOaf(entity, getDataInfo(about, provenanceAction, trust, false, false));
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			System.err.println("resultId: " + resultId);
			System.err.println("hostedbyId: " + hostedbyId);
			System.err.println("hostedbyName: " + hostedbyName);
			System.err.println("provenanceAction: " + provenanceAction);
			System.err.println("trust: " + trust);
			System.err.println("collectedFromId: " + collectedFromId);
			System.err.println("collectedFromName: " + collectedFromName);
			System.err.println("originalId: " + originalId);
			System.err.println("dateOfCollection: " + dateOfCollection);
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	private static OAIProvenance getOAIProvenance(final NodeList about) {

		OAIProvenance.Builder oaiProv = OAIProvenance.newBuilder();

		if (((about != null) && (about.getLength() > 0))) {

			final org.w3c.dom.Element provenance = getDirectChild((org.w3c.dom.Element) about.item(0), "provenance");

			if (provenance != null) {
				final org.w3c.dom.Element origDesc = getDirectChild(provenance, "originDescription");
				oaiProv.setOriginDescription(buildOriginDescription(origDesc, OriginDescription.newBuilder()));
			}
		}

		return oaiProv.build();
	}

	private static OriginDescription buildOriginDescription(final org.w3c.dom.Element origDesc, final OriginDescription.Builder od) {
		od.setHarvestDate(origDesc.getAttribute("harvestDate")).setAltered(Boolean.valueOf(origDesc.getAttribute("altered")));

		org.w3c.dom.Element elem = getDirectChild(origDesc, "baseURL");
		od.setBaseURL(elem != null ? elem.getTextContent() : "");

		elem = getDirectChild(origDesc, "identifier");
		od.setIdentifier(elem != null ? elem.getTextContent() : "");

		elem = getDirectChild(origDesc, "datestamp");
		od.setDatestamp(elem != null ? elem.getTextContent() : "");

		elem = getDirectChild(origDesc, "metadataNamespace");
		od.setMetadataNamespace(elem != null ? elem.getTextContent() : "");

		elem = getDirectChild(origDesc, "originDescription");

		if (elem != null) {

			od.setOriginDescription(buildOriginDescription(elem, OriginDescription.newBuilder()));
		}

		return od.build();
	}

}
