package eu.dnetlib.functionality.modular.ui.stats;

import java.io.StringReader;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.google.common.base.Function;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.common.logging.DnetLogger;
import eu.dnetlib.common.logging.LogMessage;
import eu.dnetlib.functionality.modular.ui.AbstractAjaxController;
import eu.dnetlib.miscutils.collections.Pair;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

@Controller
public class StatsServiceInternalController extends AbstractAjaxController {

	private final class JournalEntryFunction implements Function<Map<String, String>, StatsProcessEntry> {

		@Override
		public StatsProcessEntry apply(final Map<String, String> input) {

			final String procId = input.get(WorkflowsConstants.SYSTEM_WF_PROCESS_ID);
			final long date = NumberUtils.toLong(input.get(LogMessage.LOG_DATE_FIELD), 0);

			return new StatsProcessEntry(procId, date);
		}
	}

	private class JournalGraphEntryFunction implements Function<Map<String, String>, Map<String, Object>> {

		private String reportLogParam;
		private Map<String, String> fields;

		private final SAXReader reader = new SAXReader();
		private final SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

		public JournalGraphEntryFunction(final String reportLogParam, final List<String> list) {
			this.fields = Maps.newHashMap();
			this.reportLogParam = reportLogParam;
			for (String f : list) {
				fields.put(f.replaceAll("\\s", "").toLowerCase(), f);
			}
		}

		@Override
		public Map<String, Object> apply(final Map<String, String> input) {
			final String report = input.get(reportLogParam);
			final Map<String, Object> map = Maps.newHashMap();
			map.put("date", formatter.format(NumberUtils.toLong(input.get(LogMessage.LOG_DATE_FIELD), 0)));
			try {
				final Document doc = reader.read(new StringReader(report));
				for (Object o : doc.selectNodes("//queries")) {
					final String k = ((Element) o).valueOf("./name").replaceAll("\\s", "").toLowerCase();
					final String v = ((Element) o).valueOf("./dbResult").trim();
					if (fields.containsKey(k)) {
						map.put(fields.get(k), NumberUtils.toInt(v, 0));
					}
				}
			} catch (Exception e) {
				log.warn("Error parsing report: " + report, e);
			}
			return map;
		}
	}

	private static final Log log = LogFactory.getLog(StatsServiceInternalController.class);

	@Resource(name = "msroWorkflowLogger")
	private DnetLogger dnetLogger;

	@Value("${dnet.openaire.stats.validation.wftype}")
	private String statsWfType;

	@Value("${dnet.openaire.stats.validation.logparam}")
	private String reportLogParam;

	@Value("${dnet.openaire.stats.validation.fields}")
	private String validationFields;

	private Map<String, Object> getReportQuery() {
		final Map<String, Object> query = Maps.newHashMap();
		query.put(WorkflowsConstants.SYSTEM_WF_PROFILE_FAMILY, statsWfType);
		query.put(WorkflowsConstants.SYSTEM_COMPLETED_SUCCESSFULLY, "true");
		final Map<String, Object> queryExist = Maps.newHashMap();
		queryExist.put("$exists", true);
		query.put(reportLogParam, queryExist);
		return query;
	}

	@ResponseBody
	@RequestMapping(value = "/ui/stats/list.do")
	public List<StatsProcessEntry> list(@RequestParam(value = "start", required = true) final String start,
			@RequestParam(value = "end", required = true) final String end) throws Exception {

		final DateTimeFormatter formatter = DateTimeFormat.forPattern("yyyy-MM-dd");
		final DateTime startDate = formatter.parseDateTime(start);
		final DateTime endDate = formatter.parseDateTime(end).plusHours(23).plusMinutes(59).plusSeconds(59);

		final Map<String, Object> query = getReportQuery();

		final List<StatsProcessEntry> res = Lists.newArrayList(Iterators.transform(dnetLogger.range(startDate.toDate(), endDate.toDate(), query),
				new JournalEntryFunction()));

		log.info(String.format("found %d stat validation workflows", res.size()));

		Collections.sort(res);

		return res;
	}

	@ResponseBody
	@RequestMapping(value = "/ui/stats/getReport.do")
	public List<Pair<String, Integer>> getReport(@RequestParam(value = "procid", required = true) final String procid) throws Exception {
		final List<Pair<String, Integer>> res = Lists.newArrayList();
		Map<String, String> logEntry = dnetLogger.findOne(WorkflowsConstants.SYSTEM_WF_PROCESS_ID, procid);
		if (!logEntry.containsKey(reportLogParam)) {
			log.warn(String.format("unable to find report for procId '%s'", procid));
		} else {
			final Document report = new SAXReader().read(new StringReader(logEntry.get(reportLogParam)));
			for (Object o : report.selectNodes("//queries")) {
				final String name = ((Element) o).selectSingleNode("./name").getText().trim();
				final int count = Integer.parseInt(((Element) o).selectSingleNode("./dbResult").getText().trim());
				res.add(new Pair<String, Integer>(name, count));
			}
		}
		return res;
	}

	@ResponseBody
	@RequestMapping(value = "/ui/stats/listHistoryFields.do")
	public String listLegendFields() throws Exception {
		return validationFields;
	}

	@ResponseBody
	@RequestMapping(value = "/ui/stats/history.do")
	public List<Map<String, Object>> getHistory(@RequestParam(value = "fieldsJson", required = true) final String fieldsJson) throws Exception {
		final List<String> fields = new Gson().fromJson(fieldsJson, new TypeToken<List<String>>() {}.getType());

		final Map<String, Object> query = getReportQuery();

		final List<Map<String, Object>> res = Lists.newArrayList(Iterators.transform(dnetLogger.find(query), new JournalGraphEntryFunction(reportLogParam,
				fields)));

		log.info(String.format("found %d stat validation workflows", res.size()));
		if (log.isDebugEnabled()) {
			for (Map<String, Object> m : res) {
				for (Map.Entry<String, Object> e : m.entrySet()) {
					log.debug(e.getKey() + " -> " + e.getValue());
				}
			}
		}

		return res;
	}
}
