package eu.dnetlib.msro.openaireplus.workflows.nodes;

import java.io.StringReader;
import java.util.Map;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.SAXReader;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.resultset.ProcessCountingResultSetFactory;
import eu.dnetlib.msro.workflows.util.ProgressProvider;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;

public class UpdateRepositoryProfilesJobNode extends SimpleJobNode implements ProgressJobNode {

	private static final String REPOSITORY_SERVICE_RESOURCE_TYPE = "RepositoryServiceResourceType";

	@Resource
	private ResultSetClientFactory resultSetClientFactory;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private ProcessCountingResultSetFactory processCountingResultSetFactory;

	private String eprParam = "repoEpr";
	private String existingReposParam = "existingRepos";
	private ResultsetProgressProvider progressProvider;

	private static final Log log = LogFactory.getLog(UpdateRepositoryProfilesJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Override
	protected String execute(final NodeToken token) throws Exception {
		// datasourceID -> profileID
		@SuppressWarnings("unchecked")
		final Map<String, String> existingRepos = (Map<String, String>) token.getFullEnv().getTransientAttribute(getExistingReposParam());
		final W3CEndpointReference epr = (W3CEndpointReference) token.getFullEnv().getTransientAttribute(getEprParam());

		this.progressProvider = processCountingResultSetFactory.createProgressProvider(token.getProcess(), epr);

		final SAXReader reader = new SAXReader();

		int countUpdated = 0;
		int countInserted = 0;
		int countProfileErrors = 0;

		log.info("Adding/updating repository profiles...");

		final ISRegistryService registry = serviceLocator.getService(ISRegistryService.class);

		for (String profile : resultSetClientFactory.getClient(this.progressProvider.getEpr())) {
			try {
				final Document doc = reader.read(new StringReader(profile));
				final String dsId = doc.valueOf("//EXTRA_FIELDS/FIELD[./key='OpenAireDataSourceId']/value");

				log.debug("Registering/updating profile:\n " + profile + "\n");

				if (existingRepos.containsKey(dsId)) {
					final String profId = existingRepos.get(dsId);
					doc.selectSingleNode("//RESOURCE_IDENTIFIER/@value").setText(profId);
					registry.updateProfile(profId, doc.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
					log.debug("Profile " + profId + " UPDATED for ds " + dsId);
					countUpdated++;
				} else {
					final String profId = registry.registerProfile(doc.asXML());
					log.debug("Valid Profile " + profId + " REGISTERED for ds " + dsId);
					countInserted++;
				}
			} catch (Exception e) {
				log.error("INVALID PROFILE: " + profile, e);
				countProfileErrors++;
			}
		}

		log.info("   - updated_profiles: " + countUpdated);
		log.info("   - inserted_profiles: " + countInserted);
		log.info("   - profiles_with_errors: " + countProfileErrors);

		token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "updated_profiles", countUpdated);
		token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "inserted_profiles", countInserted);
		token.getEnv().setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + "profiles_with_errors", countProfileErrors);

		return Arc.DEFAULT_ARC;
	}

	public String getEprParam() {
		return eprParam;
	}

	public void setEprParam(final String eprParam) {
		this.eprParam = eprParam;
	}

	public String getExistingReposParam() {
		return existingReposParam;
	}

	public void setExistingReposParam(final String existingReposParam) {
		this.existingReposParam = existingReposParam;
	}

	@Override
	public ProgressProvider getProgressProvider() {
		return progressProvider;
	}

}
