package eu.dnetlib.msro.openaireplus.workflows.nodes.index;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import javax.annotation.Resource;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.ClassPathResource;

public class PrepareIndexDataJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareIndexDataJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource
	private UniqueServiceLocator serviceLocator;

	/**
	 * Stylesheet which transforms a layout to another stylesheet which converts a input record to a index record.
	 */
	private String layoutToRecordStylesheet;

	private String outputRecordsPathParam;

	private String rottenRecordsPathParam;

	private String hbaseTable;

	private String oafSchemaLocationProperty;

	@Override
	protected String execute(final NodeToken token) throws Exception {

		log.info("start preparing job");

		final String xslt = prepareXslt(env("format", token), env("layout", token));

		token.getEnv().setAttribute("index.xslt", xslt);

		if (!StringUtils.isBlank(getOutputRecordsPathParam())) {
			token.getEnv().setAttribute(getOutputRecordsPathParam(), "/tmp" + getFileName(token, "indexrecords"));
		}
		if (!StringUtils.isBlank(getRottenRecordsPathParam())) {
			token.getEnv().setAttribute(getRottenRecordsPathParam(), "/tmp" + getFileName(token, "rottenrecords"));
		}

		token.getEnv().setAttribute("index.solr.url", getIndexSolrUrlZk());
		token.getEnv().setAttribute("index.solr.collection", getCollectionName(token));

		token.getEnv().setAttribute("index.shutdown.wait.time", getIndexSolrShutdownWait());
		token.getEnv().setAttribute("index.buffer.flush.threshold", getIndexBufferFlushTreshold());
		token.getEnv().setAttribute("index.solr.sim.mode", isFeedingSimulationMode());

		token.getEnv().setAttribute("index.feed.timestamp", DateUtils.now_ISO8601());

		token.getEnv().setAttribute(getOafSchemaLocationProperty(), getPropertyFetcher().getProperty(getOafSchemaLocationProperty()));

		return Arc.DEFAULT_ARC;
	}

	protected String tableName(final NodeToken token) {
		if (token.getEnv().hasAttribute("hbaseTable")) {
			String table = token.getEnv().getAttribute("hbaseTable");
			log.debug("found override value in wfEnv for 'hbaseTable' param: " + table);
			return table;
		}
		return getHbaseTable();
	}

	public String getIndexSolrUrlZk() throws ISLookUpException {
		return getServiceConfigValue("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//PROTOCOL[./@name='solr']/@address/string()");
	}

	public String getIndexSolrShutdownWait() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingShutdownTolerance");
	}

	public String getIndexBufferFlushTreshold() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingBufferFlushThreshold");
	}

	public String isFeedingSimulationMode() throws ISLookUpException {
		return queryForServiceProperty("solr:feedingSimulationMode");
	}

	private String queryForServiceProperty(final String key) throws ISLookUpException {
		return getServiceConfigValue("for $x in /RESOURCE_PROFILE[.//RESOURCE_TYPE/@value='IndexServiceResourceType'] return $x//SERVICE_PROPERTIES/PROPERTY[./@ key='"
				+ key + "']/@value/string()");
	}

	private String getServiceConfigValue(final String xquery) throws ISLookUpException {
		log.debug("quering for service property: " + xquery);
		final String res = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
		if (StringUtils.isBlank(res)) { throw new IllegalStateException("unable to find unique service property, xquery: " + xquery); }
		return res;
	}

	private String getFileName(final NodeToken token, final String fileNamePrefix) {
		return "/" + fileNamePrefix + "_" + tableName(token) + "_" + token.getEnv().getAttribute("format") + ".seq";
	}

	private String getCollectionName(final NodeToken token) {
		return env("format", token) + "-" + env("layout", token) + "-" + env("interpretation", token);
	}

	private String env(final String s, final NodeToken token) {
		return token.getEnv().getAttribute(s);
	}

	/**
	 * Transforms each OAF record into a index record.
	 * 
	 * @param mdStoreRsetEpr
	 *            mdstore resulsetset
	 * @param layout
	 *            layout
	 * @param format
	 *            format
	 * @return resultset with transformed records
	 * @throws ISLookUpException
	 *             could happen
	 * @throws IOException
	 *             could happen
	 * @throws TransformerException
	 *             could happen
	 */
	protected String prepareXslt(final String format, final String layout) throws ISLookUpException, IOException, TransformerException {

		final TransformerFactory factory = TransformerFactory.newInstance();
		final Transformer layoutTransformer = factory.newTransformer(new StreamSource(new StringReader(readXslt(getLayoutToRecordStylesheet()))));

		final StreamResult layoutToXsltXslt = new StreamResult(new StringWriter());

		layoutTransformer.setParameter("format", format);
		layoutTransformer.transform(new StreamSource(new StringReader(getLayoutSource(format, layout))), layoutToXsltXslt);

		return new String(Hashing.encodeBase64(layoutToXsltXslt.getWriter().toString()));
	}

	private String readXslt(final String s) throws IOException {
		ClassPathResource resource = new ClassPathResource(s);
		InputStream inputStream = resource.getInputStream();
		return IOUtils.toString(inputStream);
	}

	private String getLayoutSource(final String format, final String layout) throws ISLookUpDocumentNotFoundException, ISLookUpException {
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
				"collection('')//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'MDFormatDSResourceType' and .//NAME='" + format + "']//LAYOUT[@name='" + layout
						+ "']");
	}

	public String getLayoutToRecordStylesheet() {
		return layoutToRecordStylesheet;
	}

	public void setLayoutToRecordStylesheet(final String layoutToRecordStylesheet) {
		this.layoutToRecordStylesheet = layoutToRecordStylesheet;
	}

	public String getHbaseTable() {
		return hbaseTable;
	}

	@Required
	public void setHbaseTable(final String hbaseTable) {
		this.hbaseTable = hbaseTable;
	}

	public String getOutputRecordsPathParam() {
		return outputRecordsPathParam;
	}

	public void setOutputRecordsPathParam(final String outputRecordsPathParam) {
		this.outputRecordsPathParam = outputRecordsPathParam;
	}

	public String getRottenRecordsPathParam() {
		return rottenRecordsPathParam;
	}

	public void setRottenRecordsPathParam(final String rottenRecordsPathParam) {
		this.rottenRecordsPathParam = rottenRecordsPathParam;
	}

	public String getOafSchemaLocationProperty() {
		return oafSchemaLocationProperty;
	}

	public void setOafSchemaLocationProperty(final String oafSchemaLocationProperty) {
		this.oafSchemaLocationProperty = oafSchemaLocationProperty;
	}

}
