package eu.dnetlib.pace.distance;

import java.util.List;

import eu.dnetlib.pace.model.CondDef;
import eu.dnetlib.pace.model.Document;
import eu.dnetlib.pace.model.Field;
import eu.dnetlib.pace.model.FieldDef;

/**
 * The distance between two documents is given by the weighted mean of the field distances
 */
public class DistanceScorer {

	private List<FieldDef> fields;

	private List<CondDef> conds;
	
	public DistanceScorer(List<FieldDef> fields, List<CondDef> conds) {
		this.fields = fields;
		this.conds = conds;
	}

	public double distance(Document a, Document b) {

		double w = sumWeights(fields);
		double sum = 0.0;
		boolean c = true;
		
		for(CondDef cd : conds) {
			c = c && cd.getConditionAlgo().verify(a, b);
		}

		if (c) {
			for (FieldDef fd : fields) {
				double d = fieldDistance(a, b, fd);
				if (d > 0) {
					sum += d;
				} else {
					w -= fd.getAlgo().getWeight();
				}
			}
			return w == 0 ? 0 : sum / w;
		} 
		return 0.0;		
	}

	private double fieldDistance(Document a, Document b, FieldDef fd) {
		final double w = fd.getAlgo().getWeight();
		if (w == 0) {
			return 0.0; // optimization for 0 weight
		} else {
			Field va = getValue(a, fd);
			Field vb = getValue(b, fd);

			if (va.isEmpty() || vb.isEmpty()) {
				if (fd.isIgnoreMissing()) {
					return -1;
				} else {
					return w;
				}
			} else {

				if (va.getType().equals(vb.getType())) {
					return w * fd.getAlgo().distance(va, vb);
				}
				throw new IllegalArgumentException("Types are differents type");
			}
		}
	}

	private Field getValue(Document d, FieldDef fd) {
		// TODO: Permits the use of lists
		List<Field> fields = d.values(fd.getName());
		return fields != null && !fields.isEmpty() ? fields.get(0) : new Field();
	}

	private double sumWeights(List<FieldDef> fields) {
		double sum = 0.0;
		for (FieldDef fd : fields) {
			sum += fd.getAlgo().getWeight();
		}
		return sum;
	}

}
