package eu.dnetlib.iis.ingest.pmc.citations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.codehaus.jettison.json.JSONObject;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import eu.dnetlib.iis.metadataextraction.schemas.Range;
import eu.dnetlib.iis.metadataextraction.schemas.ReferenceBasicMetadata;
import pl.edu.icm.ceon.scala_commons.xml.XPathEvaluator;
import scala.collection.JavaConversions;

/**
 * A model of resolved citation from PMC.
 *
 * @author Mateusz Fedoryszak (m.fedoryszak@icm.edu.pl)
 */
public class ResolvedCitation {
    private final String sourceOaid;
    private final int position;
    private final String rawText;
    private final Map<String, String> targetIds;

    private static final String PMID_KEY = "pmid";
    private static final String DOI_KEY = "doi";

    public ResolvedCitation(final String sourceOaid, final int position, final String rawText, final Map<String, String> targetIds) {
        this.sourceOaid = sourceOaid;
        this.position = position;
        this.rawText = rawText;
        this.targetIds = targetIds;
    }

    /**
     * @return Source (citing) document OpenAIRE ID (before dedup)
     */
    public String getSourceOaid() {
        return sourceOaid;
    }

    public int getPosition() {
        return position;
    }

    public String getRawText() {
        return rawText;
    }

    /**
     * @return Target (cited) document PubMedID
     */
    public String getTargetPmid() {
        return targetIds.get(PMID_KEY);
    }


    /**
     * @return Target (cited) document DOI
     */
    public String getTargetDoi() {
        return targetIds.get(DOI_KEY);
    }
    
    public String getTargetIdsJson() {
        return new JSONObject(targetIds).toString();
    }
    
    /**
     * Parses NLM file and extracts resolved citations.
     */
    public static List<ResolvedCitation> extractFromNlm(final String oaSourceId, final String text) {
        XPathEvaluator evaluator = XPathEvaluator.fromInputStream(IOUtils.toInputStream(text));

        List<ResolvedCitation> result = new ArrayList<ResolvedCitation>();
        int position = 1;
        for (Node ref : JavaConversions.asJavaCollection(evaluator.asNodes("/article/back/ref-list/ref"))) {
            XPathEvaluator refEvaluator = new XPathEvaluator(ref);
            Map<String, String> targetIds = new HashMap<String, String>();
            for (Node citId : JavaConversions.asJavaCollection(refEvaluator.asNodes(".//pub-id"))) {
                final String idType = citId.getAttributes().getNamedItem("pub-id-type").getTextContent();
                final String idValue = citId.getTextContent();
                targetIds.put(idType, idValue);
            }
            result.add(new ResolvedCitation(oaSourceId, position, 
            		generateReferenceRawText(ref), targetIds));

            ++position;
        }

        return result;
    }
   
    public static String generateReferenceRawText(Node ref) {
        if (containsTextChild(ref)) {
            return ref.getTextContent();
        } else {
        	return generateReferenceRawText(buildReference(ref));
        }
    }

    public static String generateReferenceRawText(ReferenceBasicMetadata refMeta) {
        String authors = refMeta.getAuthors()!=null?
        		StringUtils.join(refMeta.getAuthors(), ", "):"";
        String title = refMeta.getTitle()!=null?refMeta.getTitle().toString():null;
        String source = refMeta.getSource()!=null?refMeta.getSource().toString():null;
        String year = refMeta.getYear()!=null?refMeta.getYear().toString():null;
        String volume = refMeta.getVolume()!=null?refMeta.getVolume().toString():null;
        String issue = refMeta.getIssue()!=null?refMeta.getIssue().toString():null;
        String fpage = refMeta.getPages()!=null && refMeta.getPages().getStart()!=null
        		?refMeta.getPages().getStart().toString():null;
        String lpage = refMeta.getPages()!=null && refMeta.getPages().getEnd()!=null
        		?refMeta.getPages().getEnd().toString():null;

        StringBuilder builder = new StringBuilder();

        if (StringUtils.isNotBlank(authors)) {
            builder.append(authors);
            builder.append(". ");
        }
        if (StringUtils.isNotBlank(title)) {
            builder.append(title);
            builder.append(". ");
        }
        if (StringUtils.isNotBlank(source)) {
            builder.append(source);
            builder.append(". ");
        }
        if (StringUtils.isNotBlank(year)) {
            builder.append(year);
        }
        if (StringUtils.isNotBlank(volume)) {
            builder.append("; ");
            builder.append(volume);
        }
        if (StringUtils.isNotBlank(issue)) {
            builder.append(" (");
            builder.append(issue);
            builder.append(")");
        }
        if (StringUtils.isNotBlank(fpage)) {
            builder.append(": ");
            builder.append(fpage);
        }
        if (StringUtils.isNotBlank(lpage)) {
            builder.append("-");
            builder.append(lpage);
        }
        return builder.toString();
    }
    
    public static ReferenceBasicMetadata buildReference(Node ref) {
    	ReferenceBasicMetadata.Builder refBuilder = ReferenceBasicMetadata.newBuilder();
    	XPathEvaluator refEvaluator = new XPathEvaluator(ref);
		List<CharSequence> authorsList = new ArrayList<CharSequence>();
		for (Node name : JavaConversions.asJavaCollection(refEvaluator
				.asNodes(".//name"))) {
			XPathEvaluator nameEvaluator = new XPathEvaluator(name);
			String surname = nameEvaluator.apply(".//surname");
			String givenNames = nameEvaluator.apply(".//given-names");
			authorsList.add(surname + ", " + givenNames);
		}
		if (authorsList.size() > 0) {
			refBuilder.setAuthors(authorsList);
		}
		refBuilder.setTitle(refEvaluator.apply(".//article-title"));
		refBuilder.setSource(refEvaluator.apply(".//source"));
		refBuilder.setYear(refEvaluator.apply(".//year"));
		refBuilder.setVolume(refEvaluator.apply(".//volume"));
		refBuilder.setIssue(refEvaluator.apply(".//issue"));
		String fpage = refEvaluator.apply(".//fpage");
		String lpage = refEvaluator.apply(".//lpage");
		if (fpage!=null || lpage!=null) {
			Range.Builder rangeBuilder = Range.newBuilder();
			rangeBuilder.setStart(fpage);
			rangeBuilder.setEnd(lpage);
			refBuilder.setPages(rangeBuilder.build());
		}
		return refBuilder.build();
    }
    
    public static Boolean containsTextChild(Node ref) {
        XPathEvaluator refEval = new XPathEvaluator(ref);

        Node cit = null;
        for (String xpath : new String [] {"./citation", "./element-citation", "./mixed-citation"}) {
            cit = refEval.asNode(xpath);
            if (cit != null) {
                break;
            }
        }

        if (cit == null)
            return false;

        NodeList children = cit.getChildNodes();
        for (int i = 0; i < children.getLength(); ++i) {
            if (children.item(i).getNodeType() == Node.TEXT_NODE && 
            		!children.item(i).getNodeValue().trim().isEmpty()) {
                return true;
            }
        }
        return false;
    }
}