package eu.dnetlib.clients.data.store.ws;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.ws.BindingProvider;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.cxf.jaxws.JaxWsProxyFactoryBean;
import org.apache.log4j.Logger;

import eu.dnetlib.api.DriverServiceException;
import eu.dnetlib.api.data.StoreService;
import eu.dnetlib.api.data.StoreServiceException;
import eu.dnetlib.data.sts.ds.DepotServiceException;
import eu.dnetlib.data.sts.ds.IDepotService;
import eu.dnetlib.domain.EPR;
import eu.dnetlib.domain.ServiceIdentity;
import eu.dnetlib.domain.data.StoreInfo;
import eu.dnetlib.domain.data.StoreObjectInfo;
import eu.dnetlib.domain.enabling.Notification;
import eu.dnetlib.utils.EPRUtils;
import eu.dnetlib.utils.ServiceIdentityFactory;

public class StoreServiceWSClient implements StoreService {
	private static Logger logger = Logger.getLogger(StoreServiceWSClient.class);
	private String serviceUrl = null;
	private IDepotService service = null;
	
	public StoreServiceWSClient(String serviceUrl) {
		this.serviceUrl = serviceUrl;
		
		JaxWsProxyFactoryBean factory = new JaxWsProxyFactoryBean();
		factory.setServiceClass( eu.dnetlib.data.sts.ds.IDepotService.class);
		factory.setAddress(serviceUrl);
		this.service = (IDepotService) factory.create();
	}
	
	public StoreServiceWSClient() {
	}
	
	public void setWebService(Object webService) {
		BindingProvider prov = (BindingProvider) webService;
		
		W3CEndpointReference w3cepr = prov.getEndpointReference(W3CEndpointReference.class);
		EPR epr = EPRUtils.createEPR(w3cepr);
		
		this.serviceUrl = epr.getAddress();
		this.service = (IDepotService) webService;
		
		logger.debug("Setting serviceUrl to: " + serviceUrl);
	}

	@Override
	public ServiceIdentity identify() {
		return ServiceIdentityFactory.parseIdentity(service.identify());
	}
	
	@Override
	public void notify(Notification notification) throws DriverServiceException {
		throw new UnsupportedOperationException();
	}

	@Override
	public StoreInfo createStore(List<String> predefinedObjectTypes,
			long maxSizeStoreDS) throws StoreServiceException {
		try {
			String storeId = service.createStore(predefinedObjectTypes, maxSizeStoreDS);
			logger.debug("Created store with id: " + storeId);
			StoreInfo storeInfo = new StoreInfo();
			
			storeInfo.setServiceUrl(serviceUrl);
			storeInfo.setStoreId(storeId);
			
			return storeInfo;

		} catch (Exception e) {
			throw new StoreServiceException(e);
		}
	}

	@Override
	public void deleteStore(String storeId) throws StoreServiceException {
		try {
			logger.debug("Deleting store with id: " + storeId);
			service.deleteStore(storeId);
		} catch (Exception e) {
			throw new StoreServiceException(e);
		}
	}

	@Override
	public boolean deleteStoreObject(String storeId, List<String> storeObjectIds)
			throws StoreServiceException {
		try {
			if (logger.isDebugEnabled()) {
				logger.debug("Deleting from store " + storeId + " objects with"
						+ " ids " + Arrays.toString(storeObjectIds.toArray(new String[] {})));
			}
			
			return service.deleteStoreObject(storeId, storeObjectIds);
		} catch (Exception e) {
			throw new StoreServiceException(e);
		}
	}

	@Override
	@Deprecated
	public boolean deleteStoreObjectFromRS(String storeId, EPR rsEPR)
			throws StoreServiceException {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<StoreObjectInfo> storeObjects(String storeId,
			List<String> objectsForStoring, DataType dataType, StoringType storingType)
			throws StoreServiceException {
		try {
			List<String> storeObjects = new ArrayList<String>();
			List<StoreObjectInfo> objectInfoList = new ArrayList<StoreObjectInfo>();
			StoreInfo storeInfo = new StoreInfo();
			
			storeInfo.setServiceUrl(this.serviceUrl);
			storeInfo.setStoreId(storeId);
			
			for (String object:objectsForStoring) {
				String objectId = this.generateRandomID(object);
				StoreObjectInfo objectInfo = new StoreObjectInfo();

				if (logger.isDebugEnabled()) {
					logger.debug("Storing object " + object + " to store with"
							+ " id " + storeId + ". DataType: " + dataType
							+ ", StoringType: " + storingType +", objectId: "
							+ objectId);
				}
				
				storeObjects.add(this.getStoreRecord(object, dataType, objectId));
				
				objectInfo.setObjectId(objectId);
				objectInfo.setStoreInfo(storeInfo);
				objectInfoList.add(objectInfo);
			}
			
			String actionId = service.storeObjects(storeId, storeObjects, storingType.getValue(), false);
			
			logger.debug("Store action id: " + actionId);

			return objectInfoList;
		} catch (DepotServiceException e) {
			throw new StoreServiceException(e);
		}
	}
	
	/*
		<storeRecord>
			<storeRecordIdentifier>...</storeRecordIdentifier>
			<storeObject datatype="URI|Data">...<storeObject>
		</storeRecord>
	 */
	private String getStoreRecord(String objectForStoring, DataType dataType, String objectId) {
		StringBuilder builder = new StringBuilder();
		
		builder.append("<storeRecord><storeRecordIdentifier>");
		builder.append(objectId);
		builder.append("</storeRecordIdentifier><storeObject datatype=\"");
		builder.append(dataType.getValue());
		builder.append("\" >").append(objectForStoring);
		builder.append("</storeObject></storeRecord>");
		
		logger.debug("store record: " + builder.toString());
			
		return builder.toString();
	}
	
	private String generateRandomID(String object) {
		return System.currentTimeMillis() + "-" + object.hashCode();
	}

	@Override
	public List<StoreObjectInfo> storeObjectsFromRS(String storeId, EPR rsEPR,
			DataType dataType, StoringType storingType) throws StoreServiceException {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean updateStore(String storeId, long maxSizeStoreDS,
			List<String> predefinedObjectTypes) throws StoreServiceException {
		try {
			return service.updateStore(storeId, maxSizeStoreDS, predefinedObjectTypes);
		} catch (Exception e) {
			throw new StoreServiceException(e);
		}
	}
}
