package eu.dnetlib.openaire.user.api.services;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.unboundid.ldap.sdk.LDAPException;
import eu.dnetlib.openaire.user.LDAPUser;
import eu.dnetlib.openaire.user.MigrationUser;
import eu.dnetlib.openaire.user.Role;
import eu.dnetlib.openaire.user.dao.RoleDAO;
import eu.dnetlib.openaire.user.dao.SQLMigrationUserDAO;
import eu.dnetlib.openaire.user.ldap.MUserActionsLDAP;
import eu.dnetlib.openaire.user.security.JWTGenerator;
import eu.dnetlib.openaire.user.store.DataSourceConnector;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.sql.SQLException;

/**
 * Created by sofia on 24/11/2016.
 */
@Component
@Path("/users")
public class Test3Service {

    private static final Logger logger = Logger.getLogger(Test3Service.class);

    @Autowired
    private SQLMigrationUserDAO sqlMigrationUserDAO;

    @Autowired
    private MUserActionsLDAP mUserActionsLDAP;

    @Autowired
    private DataSourceConnector dataSourceConnector;

    @GET
    @Path("/{userId}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getUserById(@PathParam("userId") int userId) {
        try {
            MigrationUser mUser = sqlMigrationUserDAO.fetchById(userId);

            // Invalide user ID
            if (mUser == null) {
                String errorMessageJson = compose404Message("Cannot find user with id " + userId + ".");

                return Response
                        .status(Response.Status.NOT_FOUND)
                        .entity(errorMessageJson)
                        .type(MediaType.APPLICATION_JSON)
                        .build();
            }

            return Response.status(200).entity(composeDataResponse(mUser)).build();
        }
        catch (SQLException e) {
            return Response
                    .status(Response.Status.INTERNAL_SERVER_ERROR)
                    .entity(compose500Message("Fail to fetch users.", e))
                    .type(MediaType.APPLICATION_JSON)
                    .build();
        }
    }

     /* How to check @browser ../authenticate/?username=MY_USERNAME&password=MY_PASSWORD
     * http://localhost:8080/uoa-user-management-1.0.0-SNAPSHOT/api/users/authenticate?username=sba&password=12345678
    @GET
    @Path("/authenticate")
    @Produces(MediaType.APPLICATION_JSON)
    public Response authenticateUserGET(@QueryParam("username") String username, @QueryParam("password") String password)
    {
        return commonAuthenticateFunction(username, password);

    }*/

    @POST
    @Path("/authenticates")
    @Produces(MediaType.APPLICATION_JSON)
    @Consumes(MediaType.APPLICATION_JSON)
    public Response authenticateUserPOST(String input) {
        JsonObject jsonObject = new JsonParser().parse(input).getAsJsonObject();

        String username = jsonObject.get("username").getAsString();
        String password = jsonObject.get("password").getAsString();

        return commonAuthenticateFunction(username, password);
    }

    private Response commonAuthenticateFunction(String username, String password)
    {
        try {
            boolean usernameExists = mUserActionsLDAP.usernameExists(username);

            // if user was not found
            if (!usernameExists) {
                String errorMessageJson = compose401Message("Wrong credentials.");

                return Response
                        .status(Response.Status.UNAUTHORIZED)
                        .entity(errorMessageJson)
                        .type(MediaType.APPLICATION_JSON)
                        .build();
            }

            boolean authenticated = mUserActionsLDAP.authenticate(username, password);

            // if user was not authenticated
            if (!authenticated) {
                return Response
                        .status(Response.Status.UNAUTHORIZED)
                        .entity(compose401Message("User " + username + " could not be authenticated."))
                        .type(MediaType.APPLICATION_JSON)
                        .build();
            }

            MigrationUser mUser = sqlMigrationUserDAO.fetchByUsername(username);

            // if user was not found in my db
            LDAPUser ldapUser = null;
            if (mUser == null) {
                mUser = new MigrationUser(username);
                ldapUser = mUserActionsLDAP.getUser(username);
                mUser.setFullname(ldapUser.getDisplayName());
                mUser.setEmail(ldapUser.getEmail());
                mUser.setRoleId(2);


                sqlMigrationUserDAO.insert(mUser);
            }
            return Response.status(200).entity(composeDataResponse(mUser)).type(MediaType.APPLICATION_JSON).build();

        } catch (LDAPException exc) {
            logger.error("Fail to connect to LDAP. ", exc);
            return Response
                    .status(Response.Status.INTERNAL_SERVER_ERROR)
                    .entity(compose500Message("LDAP error.", exc))
                    .type(MediaType.APPLICATION_JSON)
                    .build();

        } catch (SQLException exc) {
            logger.error("Fail to fetch users. ", exc);
            return Response
                    .status(Response.Status.INTERNAL_SERVER_ERROR)
                    .entity(compose500Message("Fail to fetch users.", exc))
                    .type(MediaType.APPLICATION_JSON)
                    .build();
        }

    }

    @GET
    @Path("/changeRole")
    @Produces(MediaType.APPLICATION_JSON)
    public Response changeRole(@QueryParam("roleId") int roleId, @QueryParam("userId") int userId)
    {
        RoleDAO roleDAO = new RoleDAO();
        try
        {
            Role role = roleDAO.fetchById(roleId);
            if (role == null)
            {
                //fetch all roleids TODO
                String errorMessageJson = compose404Message("Cannot find role with id" + roleId + ".");

                return Response
                        .status(Response.Status.NOT_FOUND)
                        .entity(errorMessageJson)
                        .type(MediaType.APPLICATION_JSON)
                        .build();
            }

            MigrationUser mUser = sqlMigrationUserDAO.fetchById(userId);

            if (mUser == null)
            {
                String errorMessageJson = compose404Message("Cannot find user with id " + userId + ".");

                return Response
                        .status(Response.Status.NOT_FOUND)
                        .entity(errorMessageJson)
                        .type(MediaType.APPLICATION_JSON)
                        .build();
            }

            mUser.setRoleId(roleId);
            sqlMigrationUserDAO.update(mUser);

            return Response.status(200).entity(composeDataResponse(mUser)).build();
        }
        catch (SQLException exc)
        {
            return Response
                    .status(Response.Status.INTERNAL_SERVER_ERROR)
                    .entity(compose500Message("Fail to fetch users.", exc))
                    .type(MediaType.APPLICATION_JSON)
                    .build();
        }
    }

    /* JSON Utility Methods */

    private String compose401Message(String message) {
        return  "{ \"status\" : \"error\", \"code\" : \"401\", \"message\" : \"  " + message +" \" }";
    }

    private String compose404Message(String message) {
        return  "{ \"status\" : \"error\", \"code\" : \"404\", \"message\" : \"  " + message +" \" }";
    }

    private String compose500Message(String message, Exception exception) {
        return  "{ \"status\" : \"fail\", \"code\" : \"500\", \"message\" : \"  " + message + "\", " +
                "\"description\" : \""+  exception.getMessage() +"\" }";
    }

    private String composeDataResponse(MigrationUser user) {
        //return "{ \"status\" : \"success\", \"code\": \"200\", " + "\"data\" : " + new Gson().toJson(user) + " }";
        return "{ \"status\" : \"success\", \"code\": \"200\", " + "\"data\" : \"" + JWTGenerator.generateToken(user, "my-very-secret") + "\" }";
    }

    private String composeDataResponse(LDAPUser user) {
        return " { \"status\" : \"success\", \"code\": \"200\", " + "\"data\" : " + new Gson().toJson(user) + " }";
    }

//        private String composeDataResponse(String username) {
//            return " { \"status\" : \"success\", \"code\": \"200\", " + "\"data\" : " + new Gson().toJson(username) + " }";
//        }

    private String composeDataResponse(String fullname) {
        return " { \"status\" : \"success\", \"code\": \"200\", " + "\"data\" : " + new Gson().toJson(fullname) + " }";
    }
}
