package eu.dnetlib.broker.openaireAlerts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.broker.LiteratureBrokerServiceApplication;
import eu.dnetlib.broker.common.elasticsearch.AlertNotification;
import eu.dnetlib.broker.common.elasticsearch.Notification;
import eu.dnetlib.broker.common.properties.ElasticSearchProperties;
import eu.dnetlib.broker.common.stats.OpenaireDsStat;
import eu.dnetlib.broker.common.subscriptions.MapCondition;
import eu.dnetlib.broker.common.subscriptions.Subscription;
import eu.dnetlib.broker.objects.alerts.ValidatorAlertMessage;
import eu.dnetlib.common.controller.AbstractDnetController;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Profile("openaire")
@RestController
@RequestMapping("/api/openaire-alerts")
@Tag(name = LiteratureBrokerServiceApplication.TAG_OPENAIRE_ALERTS)
public class OpenaireAlertsBrokerController extends AbstractDnetController {

	@Autowired
	private ElasticsearchOperations esOperations;

	@Autowired
	private ElasticSearchProperties props;

	@Autowired
	private OpenaireAlertsService service;

	private static final Log log = LogFactory.getLog(OpenaireAlertsBrokerController.class);

	@Operation(summary = "Return the datasources having alerts")
	@GetMapping("/datasources-with-alerts")
	public List<DatasourceWithAlert> findDatasourcesWithAlerts() {
		return service.findDatasourcesWithAlerts();
	}

	@Operation(summary = "Perform a subscription")
	@PostMapping("/subscribe/{compatibility}")
	public Subscription registerSubscription(@PathVariable final String compatibility, @RequestParam final String email, @RequestParam final String dsId) {
		if (StringUtils.isBlank(email)) { throw new IllegalArgumentException("subscriber is empty"); }

		return service.registerSubscription(email, dsId, prepareTopicForCompatibility(compatibility));
	}

	protected static String prepareTopicForCompatibility(final String compatibility) {
		return "ALERT/" + compatibility
				.replaceAll("\\s+", "")
				.replaceAll("\\.", "_")
				.toUpperCase();
	}

	@Operation(summary = "Return the subscriptions of an user (by email and datasource (optional))")
	@GetMapping("/subscriptions")
	public List<AlertSubscriptionDesc> subscriptions(@RequestParam final String email) {
		return service.listAlertSubscriptions(email);
	}

	private String extractDatasourceId(final Subscription sub) {
		return sub.getConditionsAsList()
				.stream()
				.filter(c -> "datasourceId".equals(c.getField()))
				.map(MapCondition::getListParams)
				.filter(l -> !l.isEmpty())
				.map(l -> l.get(0).getValue())
				.findFirst()
				.orElse("");
	}

	@Operation(summary = "Return a page of alert notifications")
	@GetMapping("/notifications/{subscrId}/{nPage}/{size}")
	public AlertsPage notifications(@PathVariable final String subscrId, @PathVariable final int nPage, @PathVariable final int size) {
		final Optional<Subscription> optSub = service.findSubscription(subscrId);

		if (optSub.isPresent()) {
			final Subscription sub = optSub.get();

			final NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
					.withQuery(QueryBuilders.termQuery("subscriptionId.keyword", subscrId))
					.withSearchType(SearchType.DEFAULT)
					.withFields("payload")
					.withPageable(PageRequest.of(nPage, size))
					.build();

			final SearchHits<AlertNotification> page =
					esOperations.search(searchQuery, AlertNotification.class, IndexCoordinates.of(props.getAlertNotificationsIndexName()));

			final List<ValidatorAlertMessage> list = page.stream()
					.map(SearchHit::getContent)
					.map(Notification::getPayload)
					.map(ValidatorAlertMessage::fromJSON)
					.collect(Collectors.toList());

			return new AlertsPage(extractDatasourceId(sub), sub.getTopic(), nPage, overrideGetTotalPage(page, size), page.getTotalHits(), list);
		}
		log.warn("Invalid subscription: " + subscrId);
		return new AlertsPage("", "", nPage, 0, 0, new ArrayList<>());
	}

	@Operation(summary = "Send notifications")
	@GetMapping("/notifications/sendNotificationsForDatasource")
	private List<String> sendNotificationsForDatasource(@RequestParam final String dsId) {
		new Thread(() -> service.sendAlertNotifications(dsId)).start();
		return Arrays.asList("Sending ...");
	}

	@Operation(summary = "Update stats")
	@PostMapping("/stats/update")
	public OpenaireDsStat updateStats(@RequestBody final OpenaireDsStat stats) {
		return service.updateStats(stats);
	}

	@Operation(summary = "Delete stats for a datasource")
	@GetMapping("/stats/clearForDatasource")
	public List<OpenaireDsStat> deleteStats(@RequestParam final String dsId) {
		return service.deleteStats(dsId);
	}

	private long overrideGetTotalPage(final SearchHits<?> page, final int size) {
		return (page.getTotalHits() + size - 1) / size;
	}
}
