package eu.dnetlib.app.directindex.clients;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import eu.dnetlib.app.directindex.errors.DirectIndexApiException;
import eu.dnetlib.app.directindex.input.DatasourceEntry;

@Component
public class DatasourceManagerClient implements HasCache {

	@Value("${dnet.directindex.dsm.url}")
	private String dsmApiUrl;

	private static final Log log = LogFactory.getLog(DatasourceManagerClient.class);

	@Cacheable("datasources")
	public DatasourceEntry findDatasource(final String dsId) throws DirectIndexApiException {

		if (StringUtils.isBlank(dsId)) { return DatasourceEntry.UNKNOWN_DATASOURCE; }

		final RestTemplate rt = new RestTemplate();
		final String url = dsmApiUrl + "/searchdetails/0/1?requestSortBy=id&order=ASCENDING";
		try {
			final DsmSearchResponse res = rt.postForObject(url, new DsmSearchRequest(dsId), DsmSearchResponse.class);
			return res.getDatasourceInfo()
					.stream()
					.map(ds -> new DatasourceEntry(ds.getId(), ds.getOfficialname(), ds.getNamespaceprefix()))
					.findFirst()
					.orElse(DatasourceEntry.UNKNOWN_DATASOURCE);

		} catch (final RestClientException rce) {
			log.warn("Unable to obtain datasource: " + dsId, rce);
			throw new DirectIndexApiException("Unable to obtainn datasource: " + dsId, rce);
		}
	}

	@Override
	@CacheEvict(value = "datasources", allEntries = true)
	public void clearCache() {}

	public static class DsmDatasourceInfo implements Serializable {

		private static final long serialVersionUID = -593392381920400974L;

		private String id;
		private String officialname;
		private String namespaceprefix;

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getOfficialname() {
			return officialname;
		}

		public void setOfficialname(final String officialname) {
			this.officialname = officialname;
		}

		public String getNamespaceprefix() {
			return namespaceprefix;
		}

		public void setNamespaceprefix(final String namespaceprefix) {
			this.namespaceprefix = namespaceprefix;
		}

	}

	public static class DsmSearchRequest implements Serializable {

		private static final long serialVersionUID = -2532361140043817319L;

		private String id;

		public DsmSearchRequest() {}

		public DsmSearchRequest(final String id) {
			this.id = id;
		}

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}
	}

	public static class DsmSearchResponse implements Serializable {

		private static final long serialVersionUID = 8944122902111813747L;

		private List<DsmDatasourceInfo> datasourceInfo;

		public List<DsmDatasourceInfo> getDatasourceInfo() {
			return datasourceInfo;
		}

		public void setDatasourceInfo(final List<DsmDatasourceInfo> datasourceInfo) {
			this.datasourceInfo = datasourceInfo;
		}
	}

	public String getDsmApiUrl() {
		return dsmApiUrl;
	}

	public void setDsmApiUrl(final String dsmApiUrl) {
		this.dsmApiUrl = dsmApiUrl;
	}

}
