package eu.dnetlib.app.directindex.clients;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;

@Component
public class ProjectClient implements HasCache {

	private static final Log log = LogFactory.getLog(ProjectClient.class);

	@Value("${dnet.directindex.funders.url}")
	private String fundersApiUrl;

	@Cacheable("projects")
	public ProjectInfo resolveProjectLink(final String s) {

		// info:eu-repo/grantAgreement/EC/FP7/244909/EU/Making Capabilities Work/WorkAble

		final String[] arr = s.split("/");

		if (arr.length <= 4) { return null; }

		final ProjectInfo info = new ProjectInfo();

		final String acronym = arr.length > 7 ? arr[7] : "";
		final String title = arr.length > 6 ? StringUtils.isNotBlank(arr[6]) ? arr[6] : acronym : "";
		final String code = arr[4].replace("%2F", "/");

		final String jurisdiction = arr.length > 5 ? arr[5] : "";
		final String funderShortName = fixFunderShortName(arr[2]);

		final FunderInfo funder = findFunders().get(funderShortName.toLowerCase());

		if (funder == null) { return null; }

		final String fundingName = fixFundingName(arr[3]);
		final String fundingId = StringUtils.isNotBlank(fundingName) ? String.format("%s::%s", funder.getId(), fundingName) : null;

		final String projectPrefix = StringUtils
				.firstNonBlank(funder.getProjectPrefixByFundingName().get(fundingName.toLowerCase()), funder.getProjectPrefixByFundingName()
						.get(""), StringUtils.rightPad(funderShortName.toLowerCase(), 12, "_"));

		final String projectId = String.format("%s::%s", projectPrefix, DigestUtils.md5Hex(code));

		info.setId(projectId);
		info.setAcronym(acronym);
		info.setTitle(title);
		info.setCode(code);
		info.setJurisdiction(jurisdiction);
		info.setFunderId(funder.getId());
		info.setFunderShortName(funderShortName);
		info.setFunderName(funder.getName());
		info.setFundingId(fundingId);
		info.setFundingName(fundingName);

		return info;

	}

	@Cacheable(value = "funders", key = "'all'")
	private Map<String, FunderInfo> findFunders() {

		final Map<String, FunderInfo> map = new HashMap<>();

		map.put("aka", new FunderInfo("aka_________::AKA", "Research Council of Finland"));
		map.put("anr", new FunderInfo("anr_________::ANR", "French National Research Agency (ANR)"));
		map.put("arc", new FunderInfo("arc_________::ARC", "Australian Research Council (ARC)"));
		map.put("asap", new FunderInfo("asap________::ASAP", "Aligning Science Across Parkinson's"));
		map.put("chist-era", new FunderInfo("chistera____::CHISTERA", "CHIST-ERA"));
		map.put("cihr", new FunderInfo("cihr________::CIHR", "Canadian Institutes of Health Research"));
		map.put("conicyt", new FunderInfo("conicytf____::CONICYT", "Comisión Nacional de Investigación Científica y Tecnológica"));
		map.put("cso", new FunderInfo("501100000589::501100000589", "Chief Scientist Office, Scottish Executive"));
		map.put("dfg", new FunderInfo("dfgf________::DFG", "Deutsche Forschungsgemeinschaft"));
		map.put("dmt", new FunderInfo("501100000377::501100000377", "Dunhill Medical Trust"));
		map.put("ec", new FunderInfo("ec__________::EC", "European Commission"));
		map.put("eea", new FunderInfo("euenvagency_::EEA", "European Environment Agency"));
		map.put("epa", new FunderInfo("501100001589::501100001589", "Environmental Protection Agency"));
		map.put("fct", new FunderInfo("fct_________::FCT", "Fundação para a Ciência e a Tecnologia, I.P."));
		map.put("fwf", new FunderInfo("fwf_________::FWF", "Austrian Science Fund (FWF)"));
		map.put("gsrt", new FunderInfo("gsrt________::GSRT", "General Secretariat of Research and Technology (GSRT)"));
		map.put("hefcw", new FunderInfo("501100000383::501100000383", "Higher Education Funding Council for Wales"));
		map.put("hrzz", new FunderInfo("irb_hr______::HRZZ", "Croatian Science Foundation (CSF)"));
		map.put("inca", new FunderInfo("inca________::INCA", "Institut National du Cancer"));
		map.put("innoviris", new FunderInfo("innoviris___::INNOVIRIS", "INNOVIRIS"));
		map.put("lcs", new FunderInfo("lcs_________::LCS", "Latvian Council of Science"));
		map.put("mestd", new FunderInfo("mestd_______::MESTD", "Ministry of Education, Science and Technological Development of Republic of Serbia"));
		map.put("mnd association", new FunderInfo("501100000406::501100000406", "Motor Neuron Disease Association"));
		map.put("mzos", new FunderInfo("irb_hr______::MZOS", "Ministry of Science, Education and Sports of the Republic of Croatia (MSES)"));
		map.put("nc3rs", new FunderInfo("501100000849::501100000849", "National Centre for the Replacement, Refinement and Reduction of Animals in Research"));
		map.put("nhmrc", new FunderInfo("nhmrc_______::NHMRC", "National Health and Medical Research Council (NHMRC)"));
		map.put("nih", new FunderInfo("nih_________::NIH", "National Institutes of Health"));
		map.put("nserc", new FunderInfo("nserc_______::NSERC", "Natural Sciences and Engineering Research Council of Canada"));
		map.put("nsf", new FunderInfo("nsf_________::NSF", "National Science Foundation"));
		map.put("nwcrf", new FunderInfo("501100000357::501100000357", "North West Cancer Research Fund"));
		map.put("nwo", new FunderInfo("nwo_________::NWO", "Netherlands Organisation for Scientific Research (NWO)"));
		map.put("otka", new FunderInfo("501100008171::501100008171", "Target Ovarian Cancer"));
		map.put("rif", new FunderInfo("rif_________::RIF", "Research and Innovation Foundation"));
		map.put("rsf", new FunderInfo("rsf_________::RSF", "Russian Science Foundation"));
		map.put("sfc", new FunderInfo("501100000360::501100000360", "Scottish Funding Council"));
		map.put("sfi", new FunderInfo("sfi_________::SFI", "Science Foundation Ireland"));
		map.put("sgov", new FunderInfo("sgov________::SGOV", "Gobierno de España"));
		map.put("snsf", new FunderInfo("snsf________::SNSF", "Swiss National Science Foundation"));
		map.put("sshrc", new FunderInfo("sshrc_______::SSHRC", "Social Sciences and Humanities Research Council"));
		map.put("stfc", new FunderInfo("501100000271::501100000271", "Science and Technology Facilities Council"));
		map.put("tara", new FunderInfo("taraexp_____::tara", "Tara Expeditions Foundation"));
		map.put("tubitak", new FunderInfo("tubitakf____::tubitak", "Türkiye Bilimsel ve Teknolojik Araştırma Kurumu"));
		map.put("twcf", new FunderInfo("twcf________::TWCF", "Templeton World Charity Foundation"));
		map.put("ukri", new FunderInfo("ukri________::UKRI", "UK Research and Innovation"));
		map.put("wt", new FunderInfo("wt__________::WT", "Wellcome Trust"));

		map.forEach((k, v) -> {
			switch (k) {
			case "chist-era":
				v.getProjectPrefixByFundingName().put("", "chistera____");
				break;
			case "conicyt":
				v.getProjectPrefixByFundingName().put("", "conicytf____");
				break;
			case "dfg":
				v.getProjectPrefixByFundingName().put("", "dfgf________");
				break;
			case "ec":
				v.getProjectPrefixByFundingName().put("fp7", "corda_______");
				v.getProjectPrefixByFundingName().put("h2020", "corda__h2020");
				v.getProjectPrefixByFundingName().put("", "corda_____he");
				break;
			case "eea":
				v.getProjectPrefixByFundingName().put("", "euenvagency_");
				break;
			case "hrzz":
			case "mzos":
				v.getProjectPrefixByFundingName().put("", "irb_hr______");
				break;
			case "tara":
				v.getProjectPrefixByFundingName().put("", "taraexp_____");
				break;
			case "tubitak":
				v.getProjectPrefixByFundingName().put("", "tubitakf____");
				break;
			case "rcuk":
				v.getProjectPrefixByFundingName().put("", "ukri________");
				break;
			default:
				v.getProjectPrefixByFundingName().put("", StringUtils.rightPad(k, 12, "_"));
			}
		});

		// TODO
		// Perform call to api and ...
		// for (...) {
		// final FunderInfo funder = new FunderInfo(id, name);
		// funder.getProjectPrefixByFundingName().put(funding, prefix);
		// map.put(shortName.toLowerCase(), funder);
		// }
		return map;
	}

	@Override
	@CacheEvict(value = { "projects", "funders" }, allEntries = true)
	public void clearCache() {}

	// TODO: remove me when Zenodo will fix their data
	private String fixFunderShortName(final String funderShortName) {
		switch (funderShortName.toLowerCase()) {
		case "aff":
			return "AKA";
		case "rcuK":
			return "UKRI";
		case "rpf":
			return "RIF";
		default:
			return funderShortName;
		}
	}

	private String fixFundingName(final String fundingName) {
		if (fundingName.toLowerCase().startsWith("horizon 2020")) { return "H2020"; }
		if (fundingName.toLowerCase().startsWith("horizon europe")) { return "HE"; }
		return fundingName;
	}

	public static class FunderInfo implements Serializable {

		private static final long serialVersionUID = 4565629848874435608L;

		private final String id;

		private final String name;

		private final Map<String, String> projectPrefixByFundingName = new HashMap<String, String>();

		public FunderInfo(final String id, final String name) {
			this.id = id;
			this.name = name;
		}

		public String getId() {
			return id;
		}

		public String getName() {
			return name;
		}

		public Map<String, String> getProjectPrefixByFundingName() {
			return projectPrefixByFundingName;
		}

	}

	public static class ProjectInfo implements Serializable {

		private static final long serialVersionUID = 4433787349231982285L;

		private String id;
		private String acronym;
		private String title;
		private String code;
		private String jurisdiction;
		private String funderId;
		private String funderShortName;
		private String funderName;
		private String fundingId;
		private String fundingName;

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getAcronym() {
			return acronym;
		}

		public void setAcronym(final String acronym) {
			this.acronym = acronym;
		}

		public String getTitle() {
			return title;
		}

		public void setTitle(final String title) {
			this.title = title;
		}

		public String getCode() {
			return code;
		}

		public void setCode(final String code) {
			this.code = code;
		}

		public String getJurisdiction() {
			return jurisdiction;
		}

		public void setJurisdiction(final String jurisdiction) {
			this.jurisdiction = jurisdiction;
		}

		public String getFunderId() {
			return funderId;
		}

		public void setFunderId(final String funderId) {
			this.funderId = funderId;
		}

		public String getFunderShortName() {
			return funderShortName;
		}

		public void setFunderShortName(final String funderShortName) {
			this.funderShortName = funderShortName;
		}

		public String getFunderName() {
			return funderName;
		}

		public void setFunderName(final String funderName) {
			this.funderName = funderName;
		}

		public String getFundingId() {
			return fundingId;
		}

		public void setFundingId(final String fundingId) {
			this.fundingId = fundingId;
		}

		public String getFundingName() {
			return fundingName;
		}

		public void setFundingName(final String fundingName) {
			this.fundingName = fundingName;
		}
	}
}
