package eu.dnetlib.app.directindex.clients;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

@Component
public class CommunityClient implements HasCache {

	public static final String ZENODO_COMMUNITY = "zenodo.org/communities/";

	private static final Log log = LogFactory.getLog(CommunityClient.class);

	@Value("${dnet.directindex.community.url}")
	private String communityApiUrl;

	@Cacheable("oa_community_lists")
	public List<String> findOpenaireCommunities(final String context) {
		final String url = communityApiUrl + "/" + context + "/openairecommunities";
		return new RestTemplate().getForObject(url, ZenodoContextList.class).getOpenAirecommunitylist();
	}

	@Cacheable("contexts")
	public Map<String, ContextInfo> findNodes(final String path) {
		final String root = path.split("::")[0];

		log.debug("Preparing context: " + root);

		final Map<String, ContextInfo> res = new HashMap<>();

		final RestTemplate rt = new RestTemplate();
		final Community c = rt.getForObject(communityApiUrl + "/" + root, Community.class);

		res.put(c.getId(), new ContextInfo(StringUtils.firstNonBlank(c.getDisplayName(), c.getName()), c.getType()));

		for (final SubCommunity sc : rt.getForObject(communityApiUrl + "/" + root + "/subcommunities?all=true", SubCommunity[].class)) {
			final String[] arr = sc.getSubCommunityId().split("::");
			if (arr.length > 2) {
				res.put(arr[0] + "::" + arr[1], new ContextInfo(sc.getCategory(), ""));
				res.put(sc.getSubCommunityId(), new ContextInfo(sc.getLabel(), ""));
			}
		}

		return res
				.entrySet()
				.stream()
				.filter(e -> path.equals(e.getKey()) || path.startsWith(e.getKey() + "::"))
				.collect(Collectors.toMap(Entry::getKey, Entry::getValue));
	}

	@Override
	@CacheEvict(value = { "contexts", "oa_community_lists" }, allEntries = true)
	public void clearCache() {}

	public static class Community implements Serializable {

		private static final long serialVersionUID = 6566834038680683536L;

		private String id;
		private String type;
		private String name;
		private String shortName;
		private String displayName;
		private String displayShortName;

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getType() {
			return type;
		}

		public void setType(final String type) {
			this.type = type;
		}

		public String getName() {
			return name;
		}

		public void setName(final String name) {
			this.name = name;
		}

		public String getShortName() {
			return shortName;
		}

		public void setShortName(final String shortName) {
			this.shortName = shortName;
		}

		public String getDisplayName() {
			return displayName;
		}

		public void setDisplayName(final String displayName) {
			this.displayName = displayName;
		}

		public String getDisplayShortName() {
			return displayShortName;
		}

		public void setDisplayShortName(final String displayShortName) {
			this.displayShortName = displayShortName;
		}

	}

	public static class SubCommunity implements Serializable {

		private static final long serialVersionUID = 6363561947231890039L;

		private String subCommunityId;
		private String label;
		private String category;

		public String getSubCommunityId() {
			return subCommunityId;
		}

		public void setSubCommunityId(final String subCommunityId) {
			this.subCommunityId = subCommunityId;
		}

		public String getLabel() {
			return label;
		}

		public void setLabel(final String label) {
			this.label = label;
		}

		public String getCategory() {
			return category;
		}

		public void setCategory(final String category) {
			this.category = category;
		}

	}

	public static class ZenodoContextList implements Serializable {

		private static final long serialVersionUID = -8575901008472098218L;

		private String zenodoid;

		private List<String> openAirecommunitylist;

		public List<String> getOpenAirecommunitylist() {
			return openAirecommunitylist;
		}

		public void setOpenAirecommunitylist(final List<String> openAirecommunitylist) {
			this.openAirecommunitylist = openAirecommunitylist;

		}

		public String getZenodoid() {
			return zenodoid;
		}

		public void setZenodoid(final String zenodoid) {
			this.zenodoid = zenodoid;

		}

	}

	public static class ContextInfo implements Serializable {

		private static final long serialVersionUID = 96456546778111904L;

		private final String label;
		private final String type;

		public ContextInfo(final String label, final String type) {
			this.label = label;
			this.type = type;
		}

		public String getLabel() {
			return label;
		}

		public String getType() {
			return type;
		}

	}

	public String getCommunityApiUrl() {
		return communityApiUrl;
	}

	public void setCommunityApiUrl(final String communityApiUrl) {
		this.communityApiUrl = communityApiUrl;
	}

}
