package eu.dnetlib.app.directindex.clients;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;

@Component
public class ProjectClient implements HasCache {

	private static final Log log = LogFactory.getLog(ProjectClient.class);

	@Value("${dnet.directindex.funders.url}")
	private String fundersApiUrl;

	@Cacheable("projects")
	public ProjectInfo resolveProjectLink(final String s) {

		// info:eu-repo/grantAgreement/EC/FP7/244909/EU/Making Capabilities Work/WorkAble

		final String[] arr = s.split("/");

		if (arr.length <= 4) { return null; }

		final ProjectInfo info = new ProjectInfo();

		final String acronym = arr.length > 7 ? arr[7] : "";
		final String title = arr.length > 6 ? StringUtils.isNotBlank(arr[6]) ? arr[6] : acronym : "";
		final String code = arr[4].replace("%2F", "/");

		final String jurisdiction = arr.length > 5 ? arr[5] : "";
		final String funderShortName = fixFunderShortName(arr[2]);

		final FunderInfo funder = findFunders().get(funderShortName.toLowerCase());

		if (funder == null) { return null; }

		final String fundingName = fixFundingName(arr[3]);
		final String fundingId = StringUtils.isNotBlank(fundingName) ? String.format("%s::%s", funder.getId(), fundingName) : null;

		final String projectPrefix = StringUtils
				.firstNonBlank(funder.getProjectPrefixByFundingName().get(fundingName.toLowerCase()), funder.getProjectPrefixByFundingName()
						.get(""), StringUtils.rightPad(funderShortName.toLowerCase(), 12, "_"));

		final String projectId = String.format("%s::%s", projectPrefix, DigestUtils.md5Hex(code));

		info.setId(projectId);
		info.setAcronym(acronym);
		info.setTitle(title);
		info.setCode(code);
		info.setJurisdiction(jurisdiction);
		info.setFunderId(funder.getId());
		info.setFunderShortName(funderShortName);
		info.setFunderName(funder.getName());
		info.setFundingId(fundingId);
		info.setFundingName(fundingName);

		return info;

	}

	@Cacheable(value = "funders", key = "'all'")
	private Map<String, FunderInfo> findFunders() {

		final Map<String, FunderInfo> map = new HashMap<>();

		final RestTemplate rt = new RestTemplate();
		final FunderResponse[] funders = rt.getForObject(fundersApiUrl, FunderResponse[].class);

		for (final FunderResponse fr : funders) {
			if (StringUtils.isNoneBlank(fr.getId(), fr.getLegalName(), fr.getLegalShortName())) {

				final FunderInfo info = new FunderInfo(fr.getId(), fr.getLegalName());

				for (final FunderDatasourceResponse ds : fr.getDatasources()) {
					final String fundingProgram = StringUtils.isNotBlank(ds.getFundingProgram()) ? ds.getFundingProgram().toLowerCase() : "";
					info.getProjectPrefixByFundingName().put(fundingProgram, ds.getNsPrefix());
				}

				log.info("preparing funder info: " + fr.getId());

				map.put(fr.getLegalShortName().toLowerCase(), info);
			}
		}

		return map;
	}

	@Override
	@CacheEvict(value = { "projects", "funders" }, allEntries = true)
	public void clearCache() {}

	// TODO: remove me when Zenodo will fix their data
	private String fixFunderShortName(final String funderShortName) {
		switch (funderShortName.toLowerCase()) {
		case "aff":
			return "AKA";
		case "rcuK":
			return "UKRI";
		case "rpf":
			return "RIF";
		default:
			return funderShortName;
		}
	}

	private String fixFundingName(final String fundingName) {
		if (fundingName.toLowerCase().startsWith("horizon 2020")) { return "H2020"; }
		if (fundingName.toLowerCase().startsWith("horizon europe")) { return "HE"; }
		return fundingName;
	}

	public String getFundersApiUrl() {
		return fundersApiUrl;
	}

	public void setFundersApiUrl(final String fundersApiUrl) {
		this.fundersApiUrl = fundersApiUrl;
	}

	@JsonIgnoreProperties(ignoreUnknown = true)
	public static class FunderResponse implements Serializable {

		private static final long serialVersionUID = 7592978920486620191L;

		private String id;

		private String legalShortName;

		private String legalName;

		private List<FunderDatasourceResponse> datasources = new ArrayList<>();

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getLegalShortName() {
			return legalShortName;
		}

		public void setLegalShortName(final String legalShortName) {
			this.legalShortName = legalShortName;
		}

		public String getLegalName() {
			return legalName;
		}

		public void setLegalName(final String legalName) {
			this.legalName = legalName;
		}

		public List<FunderDatasourceResponse> getDatasources() {
			return datasources;
		}

		public void setDatasources(final List<FunderDatasourceResponse> datasources) {
			this.datasources = datasources;
		}

	}

	@JsonIgnoreProperties(ignoreUnknown = true)
	public static class FunderDatasourceResponse implements Serializable {

		private static final long serialVersionUID = -5652710486135022848L;

		private String nsPrefix;

		private String fundingProgram;

		public String getNsPrefix() {
			return nsPrefix;
		}

		public void setNsPrefix(final String nsPrefix) {
			this.nsPrefix = nsPrefix;
		}

		public String getFundingProgram() {
			return fundingProgram;
		}

		public void setFundingProgram(final String fundingProgram) {
			this.fundingProgram = fundingProgram;
		}

	}

	public static class FunderInfo implements Serializable {

		private static final long serialVersionUID = 4565629848874435608L;

		private final String id;

		private final String name;

		private final Map<String, String> projectPrefixByFundingName = new HashMap<String, String>();

		public FunderInfo(final String id, final String name) {
			this.id = id;
			this.name = name;
		}

		public String getId() {
			return id;
		}

		public String getName() {
			return name;
		}

		public Map<String, String> getProjectPrefixByFundingName() {
			return projectPrefixByFundingName;
		}

	}

	public static class ProjectInfo implements Serializable {

		private static final long serialVersionUID = 4433787349231982285L;

		private String id;
		private String acronym;
		private String title;
		private String code;
		private String jurisdiction;
		private String funderId;
		private String funderShortName;
		private String funderName;
		private String fundingId;
		private String fundingName;

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getAcronym() {
			return acronym;
		}

		public void setAcronym(final String acronym) {
			this.acronym = acronym;
		}

		public String getTitle() {
			return title;
		}

		public void setTitle(final String title) {
			this.title = title;
		}

		public String getCode() {
			return code;
		}

		public void setCode(final String code) {
			this.code = code;
		}

		public String getJurisdiction() {
			return jurisdiction;
		}

		public void setJurisdiction(final String jurisdiction) {
			this.jurisdiction = jurisdiction;
		}

		public String getFunderId() {
			return funderId;
		}

		public void setFunderId(final String funderId) {
			this.funderId = funderId;
		}

		public String getFunderShortName() {
			return funderShortName;
		}

		public void setFunderShortName(final String funderShortName) {
			this.funderShortName = funderShortName;
		}

		public String getFunderName() {
			return funderName;
		}

		public void setFunderName(final String funderName) {
			this.funderName = funderName;
		}

		public String getFundingId() {
			return fundingId;
		}

		public void setFundingId(final String fundingId) {
			this.fundingId = fundingId;
		}

		public String getFundingName() {
			return fundingName;
		}

		public void setFundingName(final String fundingName) {
			this.fundingName = fundingName;
		}
	}

}
