package eu.dnetlib.app.directindex.clients;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import eu.dnetlib.app.directindex.errors.DirectIndexApiException;

@Component
public class VocabularyClient implements HasCache {

	private static final Log log = LogFactory.getLog(VocabularyClient.class);

	@Value("${dnet.directindex.vocabulary.url}")
	private String vocApiUrl;

	@Cacheable("vocabularies")
	public Map<String, String> findVocabulary(final String voc) throws DirectIndexApiException {

		final String url = vocApiUrl + "/" + voc + ".json";

		final RestTemplate rt = new RestTemplate();
		try {
			final Map<String, String> map = new HashMap<>();
			for (final VocTerm term : rt.getForObject(url, Vocabulary.class).getTerms()) {
				map.put(term.getCode(), term.getEnglishName());
			}
			return map;
		} catch (final RestClientException e) {
			log.error("Unable to get vocabulary, url: " + url, e);
			throw new DirectIndexApiException("Unable to get vocabulary, url: " + url, e);
		}
	}

	public String findTermLabel(final String vocabulary, final String code) {
		try {
			final String label = findVocabulary(vocabulary).get(code);
			return StringUtils.firstNonBlank(label, code);
		} catch (final DirectIndexApiException e) {
			log.warn("Problem accessing vocabulary: " + vocabulary, e);
			return code;
		}
	}

	@Override
	@CacheEvict(value = "vocabularies", allEntries = true)
	public void clearCache() {}

	public static class Vocabulary implements Serializable {

		private static final long serialVersionUID = -865664758603896385L;

		private String id;
		private String name;
		private String description;
		private String code;
		private List<VocTerm> terms;

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getName() {
			return name;
		}

		public void setName(final String name) {
			this.name = name;
		}

		public String getDescription() {
			return description;
		}

		public void setDescription(final String description) {
			this.description = description;
		}

		public String getCode() {
			return code;
		}

		public void setCode(final String code) {
			this.code = code;
		}

		public List<VocTerm> getTerms() {
			return terms;
		}

		public void setTerms(final List<VocTerm> terms) {
			this.terms = terms;
		}
	}

	public static class VocTerm implements Serializable {

		private static final long serialVersionUID = 1618105318956602387L;

		private String englishName;
		private String nativeName;
		private String encoding;
		private String code;
		private List<?> synonyms;

		public String getEnglishName() {
			return englishName;
		}

		public void setEnglishName(final String englishName) {
			this.englishName = englishName;
		}

		public String getNativeName() {
			return nativeName;
		}

		public void setNativeName(final String nativeName) {
			this.nativeName = nativeName;
		}

		public String getEncoding() {
			return encoding;
		}

		public void setEncoding(final String encoding) {
			this.encoding = encoding;
		}

		public String getCode() {
			return code;
		}

		public void setCode(final String code) {
			this.code = code;
		}

		public List<?> getSynonyms() {
			return synonyms;
		}

		public void setSynonyms(final List<?> synonyms) {
			this.synonyms = synonyms;
		}
	}

	public String getVocApiUrl() {
		return vocApiUrl;
	}

	public void setVocApiUrl(final String vocApiUrl) {
		this.vocApiUrl = vocApiUrl;
	}

}
