package eu.dnetlib.app.directindex.controllers;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.app.directindex.clients.HasCache;
import eu.dnetlib.app.directindex.service.DirectIndexService;
import eu.dnetlib.app.directindex.tasks.ScheduledActions;
import io.swagger.v3.oas.annotations.Parameter;

@RestController
@RequestMapping("/api/admin")
public class AdminController {

	@Autowired
	private ScheduledActions scheduledActions;

	@Autowired
	private DirectIndexService service;

	@Autowired
	private List<HasCache> clients;

	@GetMapping("/info")
	public Map<String, ?> info() {
		return service.getInfo();
	}

	@DeleteMapping("/other-caches")
	@ResponseStatus(HttpStatus.OK)
	public void evictCaches() {
		clients.forEach(HasCache::clearCache);
	}

	@GetMapping("/scheduling/{enabled}")
	@ResponseStatus(HttpStatus.OK)
	public void updateScheduling(@PathVariable final boolean enabled) {
		scheduledActions.setEnabled(enabled);
	}

	@GetMapping("/resetExecutions/all")
	@ResponseStatus(HttpStatus.OK)
	public void resetExecutions() {
		service.resetExecutions();
	}

	@GetMapping("/resetExecutions/byCriteria")
	@ResponseStatus(HttpStatus.OK)
	public void resetExecutions(
			@Parameter(description = "The date is expressed as a UNIX timestamp (ie. number of seconds from 1970-01-01)") @RequestParam final long byTimestampAfter,
			@Parameter(description = "the ID of the datasource that provided the record") @RequestParam final String collectedFromId,
			@Parameter(description = "the type of the record (publication, dataset, software, ...) - optional") @RequestParam(required = false) final String type) {

		final LocalDateTime dt = LocalDateTime.ofInstant(Instant.ofEpochSecond(byTimestampAfter), TimeZone.getDefault().toZoneId());

		service.resetExecutions(dt, collectedFromId, type);
	}

	@GetMapping("/resetExecutions/byId")
	@ResponseStatus(HttpStatus.OK)
	public void resetExecutions(@RequestParam final String id) {
		service.resetExecutions(id);
	}

	@DeleteMapping("/records-cache/byDateBefore")
	@ResponseStatus(HttpStatus.OK)
	public void deleteExpiredRecordsByDateBefore(
			@Parameter(description = "The date is expressed as a UNIX timestamp (ie. number of seconds from 1970-01-01)") @RequestParam final long timestamp) {
		final LocalDateTime dt = LocalDateTime.ofInstant(Instant.ofEpochSecond(timestamp), TimeZone.getDefault().toZoneId());
		service.clearDatabase(dt);
	}

	@DeleteMapping("/records-cache/all")
	@ResponseStatus(HttpStatus.OK)
	public void deleteExpiredRecords() {
		service.clearDatabase();
	}

	@DeleteMapping("/records-cache/byId")
	@ResponseStatus(HttpStatus.OK)
	public void deleteExpiredRecordsById(@RequestParam final String id) {
		service.clearDatabase(id);
	}

}
