package eu.dnetlib.app.directindex.service;

import java.time.LocalDateTime;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

import javax.transaction.Transactional;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import eu.dnetlib.app.directindex.clients.DatasourceManagerClient;
import eu.dnetlib.app.directindex.errors.DirectIndexApiException;
import eu.dnetlib.app.directindex.input.DatasourceEntry;
import eu.dnetlib.app.directindex.input.ResultEntry;
import eu.dnetlib.app.directindex.repo.OperationType;
import eu.dnetlib.app.directindex.repo.PendingAction;
import eu.dnetlib.app.directindex.repo.PendingActionRepository;

@Service
public class DirectIndexService {

	private static final DatasourceEntry UNKNOWN_REPO =
			new DatasourceEntry("openaire____::1256f046-bf1f-4afc-8b47-d0b147148b18", "Unknown Repository", "unknown_____");

	@Autowired
	private PendingActionRepository pendingActionRepository;

	@Autowired
	private DatasourceManagerClient dsmClient;

	@Transactional
	public Optional<PendingAction> findPendingAction(final String id) {
		return pendingActionRepository.findById(id);
	}

	@Transactional
	public void prepareMetadataDeletion(final String openaireId, final String createdBy) {
		final PendingAction action = new PendingAction();

		action.setId(openaireId);
		action.setOperation(OperationType.DELETE);
		action.setCreatedBy(createdBy);
		action.setCreationDate(LocalDateTime.now());
		action.setExecutionDate(null);

		pendingActionRepository.save(action);
	}

	@Transactional
	public String prepareMetadataInsertOrUpdate(final ResultEntry r, final String createdBy) throws DirectIndexApiException {
		final PendingAction info = new PendingAction();

		if (StringUtils.isNotBlank(r.getOpenaireId())) {
			if (!r.getOpenaireId().matches("^\\w{12}::\\w{32}$")) {
				throw new DirectIndexApiException("Invalid openaireId: " + r.getOpenaireId() + " - regex ^\\w{12}::\\w{32}$ not matched");
			}
			info.setOperation(OperationType.UPDATE);
		} else if (StringUtils.isNoneBlank(r.getOriginalId(), r.getCollectedFromId())) {
			fixOpenaireId(r);
			info.setOperation(OperationType.INSERT);
		} else {
			throw new DirectIndexApiException("Missing identifier fields: [openaireId] or [originalId, collectedFromId]");
		}

		if (StringUtils.isBlank(r.getTitle())) { throw new DirectIndexApiException("A required field is missing: title"); }
		if (StringUtils.isBlank(r.getUrl())) { throw new DirectIndexApiException("A required field is missing: url"); }
		if (StringUtils.isBlank(r.getAccessRightCode())) { throw new DirectIndexApiException("A required field is missing: accessRightCode"); }
		if (StringUtils.isBlank(r.getResourceType())) { throw new DirectIndexApiException("A required field is missing: resourceType"); }
		if (StringUtils.isBlank(r.getCollectedFromId())) { throw new DirectIndexApiException("A required field is missing: collectedFromId"); }
		if (StringUtils.isBlank(r.getType())) { throw new DirectIndexApiException("A required field is missing: type"); }

		info.setId(r.getOpenaireId());
		info.setBody(r.toJson());
		info.setType(r.getType());
		info.setCreatedBy(createdBy);
		info.setCreationDate(LocalDateTime.now());
		info.setExecutionDate(null);

		pendingActionRepository.save(info);

		return info.getId();
	}

	private void fixOpenaireId(final ResultEntry r) throws DirectIndexApiException {
		final DatasourceEntry ds = StringUtils.isNotBlank(r.getCollectedFromId()) ? dsmClient.findDatasource(r.getCollectedFromId()) : UNKNOWN_REPO;
		final String openaireId = ds.getPrefix() + "::" + DigestUtils.md5Hex(r.getOriginalId());
		r.setOpenaireId(openaireId);
	}

	@Transactional
	public void clearDatabase() {
		pendingActionRepository.deleteAll();
	}

	@Transactional
	public void clearDatabase(final String id) {
		pendingActionRepository.deleteById(id);
	}

	@Transactional
	public void clearDatabase(final LocalDateTime datetime) {
		pendingActionRepository.deleteByCreationDateBefore(datetime);
	}

	@Transactional
	public void resetExecutions() {
		pendingActionRepository.resetAllExecutions();
	}

	@Transactional
	public void resetExecutions(final LocalDateTime byDatetimeAfter, final String collectedFromId, final String type) {
		if (StringUtils.isBlank(type)) {
			pendingActionRepository.resetExecutionsForAllTypes(byDatetimeAfter, collectedFromId);
		} else {
			pendingActionRepository.resetExecutionsForType(byDatetimeAfter, collectedFromId, type);
		}
	}

	@Transactional
	public void resetExecutions(final String id) {
		pendingActionRepository.resetExecutionForId(id);
	}

	@Transactional
	public Map<String, ?> getInfo() {
		final Map<String, Object> info = new LinkedHashMap<String, Object>();

		info.put("total_actions", pendingActionRepository.count());
		info.put("total_actions_insert", pendingActionRepository.countByOperation(OperationType.INSERT));
		info.put("total_actions_update", pendingActionRepository.countByOperation(OperationType.UPDATE));
		info.put("total_actions_delete", pendingActionRepository.countByOperation(OperationType.DELETE));

		info.put("pending_actions", pendingActionRepository.countByExecutionDateIsNull());
		info.put("pending_actions_insert", pendingActionRepository.countByOperationAndExecutionDateIsNull(OperationType.INSERT));
		info.put("pending_actions_update", pendingActionRepository.countByOperationAndExecutionDateIsNull(OperationType.UPDATE));
		info.put("pending_actions_delete", pendingActionRepository.countByOperationAndExecutionDateIsNull(OperationType.DELETE));

		info.put("min_registration_date", pendingActionRepository.findMinCreationDate());
		info.put("max_registration_date", pendingActionRepository.findMaxCreationDate());
		info.put("min_execution_date", pendingActionRepository.findMinExecutionDate());
		info.put("max_execution_date", pendingActionRepository.findMaxExecutionDate());

		info.put("errors", pendingActionRepository.countByErrorIsNotNull());

		return info;
	}

}
