package eu.dnetlib.app.directindex.tasks;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.common.SolrInputDocument;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;

import eu.dnetlib.app.directindex.input.ResultEntry;
import eu.dnetlib.app.directindex.mapping.SolrRecordMapper;
import eu.dnetlib.app.directindex.repo.PendingAction;
import eu.dnetlib.app.directindex.repo.PendingActionRepository;
import eu.dnetlib.app.directindex.solr.SolrIndexClient;
import eu.dnetlib.app.directindex.solr.SolrIndexClientFactory;

@Component
public class ScheduledActions {

	private static final Log log = LogFactory.getLog(ScheduledActions.class);

	@Value("${dnet.directindex.scheduling.enabled}")
	private boolean enabled;

	@Autowired
	private SolrIndexClientFactory solrIndexClientFactory;

	@Autowired
	private SolrRecordMapper solrRecordMapper;

	@Autowired
	private PendingActionRepository pendingActionRepository;

	@Value("${dnet.directindex.scheduling.nThreads}")
	private int nThreads;

	@Value("${dnet.directindex.scheduling.maxActionsForThread}")
	private int maxActionsForThread;

	@Value("${dnet.directindex.maxNumberOfDays}")
	private int maxNumberOfDays;

	@Scheduled(initialDelay = 1, fixedDelayString = "${dnet.directindex.scheduling.upsert.interval}", timeUnit = TimeUnit.MINUTES)
	public synchronized void indexRecords() throws InterruptedException {

		if (!enabled) {
			log.debug("SKIP");
			return;
		}

		final Instant start = Instant.now();

		final List<PendingAction> list = pendingActionRepository.findInsertOrUpdateOperations();

		if (list.isEmpty()) { return; }

		log.debug("Start Indexing new records, size=" + list.size());

		final ExecutorService executor = Executors.newFixedThreadPool(nThreads);
		Lists.partition(list, maxActionsForThread).forEach(subList -> executor.execute(() -> indexRecords(subList)));
		executor.shutdown();

		if (executor.awaitTermination(20, TimeUnit.MINUTES)) {
			final long timeElapsed = Duration.between(start, Instant.now()).toSeconds() + 1; // I ADD 1 TO AVOID DIVISION BY 0
			log.info(String.format("Indexed %s records in %d seconds (%.3f records/s)", list.size(), timeElapsed, (float) list.size() / timeElapsed));
		} else {
			log.warn("Some threads continue to be running");
		}
	}

	private void indexRecords(final List<PendingAction> list) {

		log.debug("(THREAD) Start indexing...");
		try {
			final Instant start = Instant.now();

			final Map<String, String> invalids = new HashMap<>();

			final SolrIndexClient solr = solrIndexClientFactory.getClient();

			final ObjectMapper objectMapper = new ObjectMapper();

			final Iterator<SolrInputDocument> iter = list.stream()
					.map(pendingAction -> {
						final String id = pendingAction.getId();
						final String body = pendingAction.getBody();

						try {
							final ResultEntry resultEntry = objectMapper.readValue(body, ResultEntry.class);
							final SolrInputDocument doc = solrRecordMapper.toSolrInputRecord(resultEntry);
							return doc;
						} catch (final Throwable e) {
							invalids.put(id, e.getClass().getName() + ": " + e.getMessage());
							log.error(e);
							return null;
						}
					})
					.filter(Objects::nonNull)
					.iterator();

			solr.addRecords(iter);
			solr.commit();

			updateExecutionDate(list, invalids);

			final long timeElapsed = Duration.between(start, Instant.now()).toSeconds() + 1; // I ADD 1 TO AVOID DIVISION BY 0

			log.debug(String.format("(THREAD) Indexed %s records in %d seconds (%.3f records/s)", list.size(), timeElapsed, (float) list.size() / timeElapsed));

		} catch (final Throwable e) {
			log.error("(THREAD) Error indexing records", e);
		}
	}

	@Scheduled(initialDelay = 5, fixedDelayString = "${dnet.directindex.scheduling.delete.interval}", timeUnit = TimeUnit.MINUTES)
	public synchronized void deleteRecords() {

		if (!enabled) {
			log.debug("SKIP");
			return;
		}

		try {

			final List<PendingAction> list = pendingActionRepository.findDeleteOperations();
			final Map<String, String> invalids = new HashMap<>();

			if (list.size() > 0) {
				log.debug("Start deleting records from index, size=" + list.size());

				final SolrIndexClient solr = solrIndexClientFactory.getClient();

				list.stream().map(PendingAction::getId).forEach(id -> {
					try {
						solr.deleteRecord(id);
					} catch (final Throwable e) {
						invalids.put(id, e.getMessage());
						log.error(e);
					}
				});

				solr.commit();

				updateExecutionDate(list, invalids);

				log.info(String.format("Deleted records: %s", list.size()));
			}
		} catch (final Throwable e) {
			log.error("The scheduled task is failed", e);
		}
	}

	@Scheduled(initialDelay = 1, fixedDelay = 24, timeUnit = TimeUnit.HOURS)
	public synchronized void deleteObsoleteRecords() throws InterruptedException {
		log.info("Deleting obsoleted records from the cache");
		pendingActionRepository.deleteObsoleteRecords(maxNumberOfDays);
	}

	private void updateExecutionDate(final List<PendingAction> list, final Map<String, String> invalids) {
		final LocalDateTime now = LocalDateTime.now();
		list.forEach(r -> {
			r.setError(invalids.getOrDefault(r.getId(), null));
			r.setExecutionDate(now);
		});
		pendingActionRepository.saveAll(list);
	}

	public boolean isEnabled() {
		return enabled;
	}

	public synchronized void setEnabled(final boolean enabled) {
		this.enabled = enabled;
	}

}
