package eu.dnetlib.app.directindex.mapping;

import org.apache.commons.lang3.StringUtils;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.QName;

import eu.dnetlib.dhp.schema.common.ModelConstants;
import eu.dnetlib.dhp.schema.solr.BestAccessRight;
import eu.dnetlib.dhp.schema.solr.Instance;
import eu.dnetlib.dhp.schema.solr.Language;
import eu.dnetlib.dhp.schema.solr.RecordType;
import eu.dnetlib.dhp.schema.solr.RelatedRecord;
import eu.dnetlib.dhp.schema.solr.SolrRecord;

public class XMLSolrSerializer {

	private static final Namespace DRI_NS = new Namespace("dri", "http://www.driver-repository.eu/namespace/dri");
	private static final Namespace OAF_NS = new Namespace("oaf", "http://namespace.openaire.eu/oaf");
	private static final Namespace XSI_NS = new Namespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");

	public static String generateXML(final SolrRecord sr) {
		final Element result = DocumentHelper.createElement("result");

		populateHeader(sr, result.addElement("header"));
		poulateMetadata(sr, result.addElement("metadata"));

		return result.asXML();
	}

	private static void populateHeader(final SolrRecord sr, final Element header) {
		header.addElement(new QName("objIdentifier", DRI_NS)).addText(sr.getHeader().getId());
		header.addElement(new QName("dateOfCollection", DRI_NS)).addText("");
		header.addElement(new QName("dateOfTransformation", DRI_NS)).addText("");
		header.addElement(new QName("status", DRI_NS)).addText(sr.getHeader().getStatus().name());
	}

	private static void poulateMetadata(final SolrRecord sr, final Element metadata) {
		final Element entity = metadata.addElement(new QName("entity", OAF_NS));
		entity.addAttribute(new QName("schemaLocation", XSI_NS), "http://namespace.openaire.eu/oaf https://www.openaire.eu/schema/1.0/oaf-1.0.xsd");
		if (sr.getResult() != null) {
			populateResultFields(sr, entity.addElement(new QName("result", OAF_NS)));
		}
	}

	@SuppressWarnings("deprecation")
	private static void populateResultFields(final SolrRecord sr, final Element fields) {

		if (sr.getHeader().getOriginalId() != null) {
			sr.getHeader().getOriginalId().forEach(id -> fields.addElement("originalId").addText(id));
		}

		addStructuredField(fields, "title", "main title", ModelConstants.DNET_DATACITE_TITLE, sr.getResult().getMaintitle());

		if (sr.getResult().getBestaccessright() != null) {
			final BestAccessRight r = sr.getResult().getBestaccessright();
			addStructuredField(fields, "bestaccessright", r.getCode(), r.getLabel(), ModelConstants.DNET_ACCESS_MODES, null);
		}

		if (sr.getResult().getAuthor() != null) {
			sr.getResult().getAuthor().forEach(a -> {
				final Element creator = fields.addElement("creator");
				creator.addAttribute("rank", "" + a.getRank());
				creator.addText(a.getFullname());
			});
		}

		if (sr.getResult().getDescription() != null) {
			sr.getResult().getDescription().forEach(s -> fields.addElement("description").addText(s));
		}

		if (sr.getResult().getLanguage() != null) {
			final Language l = sr.getResult().getLanguage();
			addStructuredField(fields, "language", l.getCode(), l.getLabel(), ModelConstants.DNET_LANGUAGES, null);
		}

		if (sr.getResult().getResulttype() != null) {
			addStructuredField(fields, "resulttype", sr.getResult().getResulttype(), ModelConstants.DNET_RESULT_TYPOLOGIES, null);
		}

		if (sr.getContext() != null) {
			sr.getContext().forEach(ctx -> {
				final Element ctxNode = fields.addElement("context");
				ctxNode.addAttribute("id", ctx.getId());
				ctxNode.addAttribute("label", ctx.getLabel());
				ctxNode.addAttribute("type", ctx.getType());
				if (ctx.getCategory() != null) {
					ctx.getCategory().forEach(cat -> {
						final Element catNode = ctxNode.addElement("category");
						catNode.addAttribute("id", cat.getId());
						catNode.addAttribute("label", cat.getLabel());
						if (cat.getConcept() != null) {
							cat.getConcept().forEach(cpt -> {
								final Element cptNode = catNode.addElement("concept");
								cptNode.addAttribute("id", cpt.getId());
								cptNode.addAttribute("label", cpt.getLabel());
							});
						}
					});
				}
			});
		}

		if (sr.getResult().getEmbargoenddate() != null) {
			fields.addElement("embargoenddate").addText(sr.getResult().getEmbargoenddate());
		}

		if (sr.getCollectedfrom() != null) {
			sr.getCollectedfrom().forEach(p -> {
				final Element collectedFrom = fields.addElement("collectedfrom");
				collectedFrom.addAttribute("id", p.getDsId());
				collectedFrom.addAttribute("name", p.getDsName());
			});
		}

		final Element datainfo = fields.addElement("datainfo");
		datainfo.addElement("inferred").addText("false");
		datainfo.addElement("deletedbyinference").addText("false");
		datainfo.addElement("trust").addText("0.9");
		datainfo.addElement("inferenceprovenance").addText("");
		addStructuredField(datainfo, "provenanceaction", "user:insert", ModelConstants.DNET_PROVENANCE_ACTIONS, null);

		final Element relsNode = fields.addElement("rels");
		if (sr.getLinks() != null) {
			sr.getLinks().forEach(p -> addRelProject(relsNode, p));
		}

		final Element childrenNode = fields.addElement("children");
		if (sr.getResult().getInstance() != null) {
			sr.getResult().getInstance().forEach(p -> addResultInstance(childrenNode, p));
		}
	}

	private static void addResultInstance(final Element childrenNode, final Instance i) {
		final Element instanceNode = childrenNode.addElement("instance");

		if (i instanceof InstanceWithTypeCode) {
			addStructuredField(instanceNode, "instancetype", ((InstanceWithTypeCode) i).getInstancetypeCode(), i
					.getInstancetype(), ModelConstants.DNET_PUBLICATION_RESOURCE, null);
		}

		addStructuredField(instanceNode, "accessright", i.getAccessright().getCode(), i.getAccessright().getLabel(), ModelConstants.DNET_ACCESS_MODES, null);

		final Element collectedFrom = instanceNode.addElement("collectedfrom");
		collectedFrom.addAttribute("id", i.getCollectedfrom().getDsId());
		collectedFrom.addAttribute("name", i.getCollectedfrom().getDsName());

		final Element hostedBy = instanceNode.addElement("hostedby");
		hostedBy.addAttribute("id", i.getHostedby().getDsId());
		hostedBy.addAttribute("name", i.getHostedby().getDsName());

		if (i.getUrl() != null) {
			i.getUrl().forEach(url -> instanceNode.addElement("webresource").addElement("url").addText(url));
		}

	}

	private static void addRelProject(final Element relsNode, final RelatedRecord p) {

		if (p.getHeader().getRelatedRecordType() != RecordType.project) { return; }

		final Element rel = relsNode.addElement("rel");

		rel.addAttribute("inferred", "false");
		rel.addAttribute("trust", "0.9");
		rel.addAttribute("inferenceprovenance", "");
		rel.addAttribute("provenanceaction", ModelConstants.USER_CLAIM);

		final Element to = rel.addElement("to");
		to.addAttribute("class", ModelConstants.IS_PRODUCED_BY);
		to.addAttribute("scheme", "dnet:result_project_relations");
		to.addAttribute("type", RecordType.project.name());
		to.addText(p.getHeader().getRelatedIdentifier());

		rel.addElement("code").addText(p.getCode());
		rel.addElement("acronym").addText(p.getAcronym());
		rel.addElement("title").addText(p.getProjectTitle());

		if (p.getFunding() != null) {
			final Element funding = rel.addElement("funding");

			if (p.getFunding().getFunder() != null) {
				final Element funder = funding.addElement("funder");
				funder.addAttribute("id", p.getFunding().getFunder().getId());
				funder.addAttribute("name", p.getFunding().getFunder().getName());
				funder.addAttribute("shortname", p.getFunding().getFunder().getShortname());
				if (p.getFunding().getFunder().getJurisdiction() != null) {
					funder.addAttribute("jurisdiction", p.getFunding().getFunder().getJurisdiction().getCode());
				}
			}

			if (p.getFunding().getLevel0() != null) {
				final Element level0 = funding.addElement("funding_level_0");
				level0.addAttribute("name", p.getFunding().getLevel0().getName());
				level0.addText(p.getFunding().getLevel0().getId());
			}
		}
	}

	private static void addStructuredField(final Element parent, final String fieldName, final String classid, final String scheme, final String value) {
		addStructuredField(parent, fieldName, classid, classid, scheme, value);
	}

	private static void addStructuredField(final Element parent,
			final String fieldName,
			final String classid,
			final String classname,
			final String scheme,
			final String value) {
		final Element elem = parent.addElement(fieldName);
		elem.addAttribute("classid", classid);
		elem.addAttribute("classname", classname);
		elem.addAttribute("schemeid", scheme);
		elem.addAttribute("schemename", scheme);

		if (StringUtils.isNotBlank(value)) {
			elem.addText(value);
		}
	}

}
