BEGIN;

DROP VIEW IF EXISTS oa_duplicates_view;
DROP VIEW IF EXISTS organizations_view;
DROP VIEW IF EXISTS organizations_info_view;
DROP VIEW IF EXISTS organizations_simple_view;
DROP VIEW IF EXISTS users_view;
DROP VIEW IF EXISTS conflict_groups_view;
DROP VIEW IF EXISTS suggestions_info_by_country_view;
DROP VIEW IF EXISTS duplicate_groups_view;
DROP VIEW IF EXISTS relationship_view;
DROP VIEW IF EXISTS relationship_groups_view;
DROP VIEW IF EXISTS persistent_orgs_view;
DROP VIEW IF EXISTS api_journal_view;

CREATE VIEW oa_duplicates_view AS SELECT
	d.local_id                                  as local_id,
	d.oa_original_id                            as oa_original_id,
	o.name                                      as oa_name, 
	array_to_string(array_agg(DISTINCT a.acronym), ', ') as oa_acronym,
	o.country                                   as oa_country,
	array_to_string(array_agg(DISTINCT u.url), ', ')     as oa_url,
	COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', oid.otherid, 'type', oid.type)) FILTER (WHERE oid.otherid IS NOT NULL), '[]') AS oa_other_ids,
	d.oa_collectedfrom                          as oa_collectedfrom,
	d.status                                    as status,
	d.created_by                                as created_by,
	o.ec_legalbody,
	o.ec_legalperson,
	o.ec_nonprofit,
	o.ec_researchorganization,
	o.ec_highereducation,
	o.ec_internationalorganizationeurinterests,
	o.ec_internationalorganization,
	o.ec_enterprise,
	o.ec_smevalidated,
	o.ec_nutscode,
	substr(d.oa_original_id, 1, 14)||md5(substr(d.oa_original_id, 15)) as openaire_id,
	false as openaire_persistent
FROM
	oa_duplicates d
	LEFT OUTER JOIN organizations o ON (o.id = d.oa_original_id)
	LEFT OUTER JOIN acronyms a      ON (o.id = a.id)
	LEFT OUTER JOIN urls u          ON (o.id = u.id)
	LEFT OUTER JOIN other_ids oid   ON (o.id = oid.id)
GROUP BY
	d.local_id,
	d.oa_original_id,
	d.created_by, 
	o.name,
	o.country,
	d.oa_collectedfrom,
	d.status,
	o.ec_legalbody,
	o.ec_legalperson,
	o.ec_nonprofit,
	o.ec_researchorganization,
	o.ec_highereducation,
	o.ec_internationalorganizationeurinterests,
	o.ec_internationalorganization,
	o.ec_enterprise,
	o.ec_smevalidated,
	o.ec_nutscode;

CREATE VIEW organizations_view AS SELECT
    org.id,
    org.name,
    org.type,
    org.lat,
	org.lng,
	org.city,
	org.country,
	org.status,
	org.ec_legalbody,
	org.ec_legalperson,
	org.ec_nonprofit,
	org.ec_researchorganization,
	org.ec_highereducation,
	org.ec_internationalorganizationeurinterests,
	org.ec_internationalorganization,
	org.ec_enterprise,
	org.ec_smevalidated,
	org.ec_nutscode,
	substr(org.id, 1, 14)||md5(substr(org.id, 15)) as openaire_id,
	count(po.id) > 0 as openaire_persistent,
	COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', oid.otherid, 'type', oid.type))                                         FILTER (WHERE oid.otherid IS NOT NULL), '[]') AS other_ids,
	COALESCE(jsonb_agg(DISTINCT jsonb_build_object('name', n.name, 'lang', n.lang))                                              FILTER (WHERE n.name      IS NOT NULL), '[]') AS other_names,
	COALESCE(jsonb_agg(DISTINCT a.acronym)                                                                                       FILTER (WHERE a.acronym   IS NOT NULL), '[]') AS acronyms,	
	COALESCE(jsonb_agg(DISTINCT u.url)                                                                                           FILTER (WHERE u.url       IS NOT NULL), '[]') AS urls
FROM
    organizations org
    LEFT OUTER JOIN other_ids oid         ON (org.id = oid.id)
    LEFT OUTER JOIN other_names n         ON (org.id = n.id)
    LEFT OUTER JOIN acronyms a            ON (org.id = a.id)
    LEFT OUTER JOIN urls u                ON (org.id = u.id)
	LEFT OUTER JOIN persistent_orgs po    ON (org.id = po.id)
GROUP BY
    org.id,
    org.name,
    org.type,
    org.lat,
	org.lng,
	org.city,
	org.country,
	org.status,
	org.ec_legalbody,
	org.ec_legalperson,
	org.ec_nonprofit,
	org.ec_researchorganization,
	org.ec_highereducation,
	org.ec_internationalorganizationeurinterests,
	org.ec_internationalorganization,
	org.ec_enterprise,
	org.ec_smevalidated,
	org.ec_nutscode;
	
CREATE VIEW organizations_info_view AS SELECT
	org.id,
	org.name,
	org.created_by,
	org.creation_date,
	org.modified_by,
	org.modification_date,
	substr(org.id, 1, 14)||md5(substr(org.id, 15)) as openaire_id,
	count(po.id) > 0 as openaire_persistent,
	count(DISTINCT d.oa_original_id) as n_duplicates,
	count(DISTINCT c.id2) as n_conflicts,
	count(DISTINCT r.id2) AS n_relations,
	count(DISTINCT n.note) > 0 as note
FROM organizations org
	LEFT OUTER JOIN oa_duplicates d    ON (org.id = d.local_id AND d.status = 'suggested')
	LEFT OUTER JOIN oa_conflicts  c    ON (org.id = c.id1 AND c.status = 'suggested')
	LEFT OUTER JOIN relationships r    ON (org.id = r.id1 AND r.status = 'suggested')
	LEFT OUTER JOIN notes n            ON (org.id = n.id)
	LEFT OUTER JOIN persistent_orgs po ON (org.id = po.id)
GROUP BY org.id;

CREATE VIEW users_view AS SELECT
	u.email,
	u.fullname,
	u.organization,
	u.reference_person,
	u.request_message,
	u.valid,
	u.role,
	u.first_access,
	u.last_access,
	array_remove(array_agg(uc.country), NULL) AS countries
FROM
	users u
	LEFT OUTER JOIN user_countries uc ON (u.email = uc.email)
GROUP BY 
	u.email,
	u.fullname,
	u.organization,
	u.reference_person,
	u.request_message,
	u.valid,
	u.role,
	u.first_access,
	u.last_access
ORDER BY u.email;

CREATE VIEW suggestions_info_by_country_view AS SELECT 
	c.val                          AS code,
	c.name                         AS name,
	coalesce(t1.n_duplicates,   0) AS n_duplicates,
	coalesce(t2.n_conflicts,    0) AS n_conflicts,
	coalesce(t3.n_relations,    0) AS n_relations,
	coalesce(t4.n_pending_orgs, 0) AS n_pending_orgs
FROM countries c
	LEFT OUTER JOIN (SELECT o.country AS country, count(DISTINCT d.local_id) AS n_duplicates   FROM oa_duplicates d LEFT OUTER JOIN organizations o ON (d.local_id = o.id) WHERE d.status = 'suggested' AND o.status = 'approved' GROUP BY o.country) AS t1 ON (t1.country = c.val)
	LEFT OUTER JOIN (SELECT o.country AS country, count(DISTINCT c.idgroup)  AS n_conflicts    FROM oa_conflicts  c LEFT OUTER JOIN organizations o ON (c.id1 = o.id)      WHERE c.status = 'suggested' AND o.status = 'approved' GROUP BY o.country) AS t2 ON (t2.country = c.val)
	LEFT OUTER JOIN (SELECT o.country AS country, count(DISTINCT r.id1)      AS n_relations    FROM relationships r LEFT OUTER JOIN organizations o ON (r.id1 = o.id)      WHERE r.status  = 'suggested' AND o.status = 'approved' AND (r.reltype = 'IsChildOf' OR r.reltype = 'IsParentOf') GROUP BY o.country) AS t3 ON (t3.country = c.val)
	LEFT OUTER JOIN (SELECT o.country AS country, count(DISTINCT o.id)       AS n_pending_orgs FROM organizations o WHERE o.status = 'suggested' GROUP BY o.country) AS t4 ON (t4.country = c.val)
ORDER BY c.name;

CREATE VIEW conflict_groups_view AS SELECT
    c.idgroup  AS idgroup,
	o1.id      AS id_1,
    o1.name    AS name_1,
    o1.type    AS type_1,
	o1.city    AS city_1,
	o1.country AS country_1,
    o2.id      AS id_2,
    o2.name    AS name_2,
    o2.type    AS type_2,
	o2.city    AS city_2,
	o2.country AS country_2
FROM
	oa_conflicts c
	LEFT OUTER JOIN organizations o1 ON (c.id1 = o1.id)
    LEFT OUTER JOIN organizations o2 ON (c.id2 = o2.id)
WHERE
	o1.id IS NOT NULL 
	AND o2.id IS NOT NULL 
	AND o1.status = 'approved'
	AND o2.status = 'approved'
	AND c.idgroup IS NOT NULL 
	AND c.status = 'suggested';
	
CREATE VIEW duplicate_groups_view AS SELECT
	o.id,
	o.name,
	o.city,
	o.country,
	count(d.*) as n_duplicates
FROM
	oa_duplicates d
	LEFT OUTER JOIN organizations o ON (o.id = d.local_id)
WHERE
	d.status = 'suggested' AND o.status = 'approved'
GROUP BY o.id, o.name, o.city, o.country
ORDER BY o.name;

CREATE VIEW relationship_view AS SELECT
	r.id1                AS id1,
	r.id2                AS id2,
	substr(r.id2, 1, 14)||md5(substr(r.id2,15)) as openaire_id2,
	r.reltype            AS reltype,
	r.status             AS status,
	o.name               AS name,
	o.city               AS city,
	o.country            AS country,
	COALESCE(jsonb_agg(DISTINCT a.acronym) FILTER (WHERE a.acronym   IS NOT NULL), '[]') AS acronyms,	
	COALESCE(jsonb_agg(DISTINCT u.url)     FILTER (WHERE u.url       IS NOT NULL), '[]') AS urls,
	COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', oid.otherid, 'type', oid.type)) FILTER (WHERE oid.otherid IS NOT NULL), '[]') AS other_ids
FROM
	relationships r
	JOIN organizations o ON (r.id2 = o.id)
	LEFT OUTER JOIN other_ids oid         ON (o.id = oid.id)
    LEFT OUTER JOIN acronyms a            ON (o.id = a.id)
    LEFT OUTER JOIN urls u                ON (o.id = u.id)
GROUP BY
	r.id1,
	r.id2,
	r.reltype,
	r.status,
	o.name,
	o.city,
	o.country
ORDER BY r.reltype, o.name;

CREATE VIEW relationship_groups_view AS SELECT
	o.id,
	o.name,
	o.city,
	o.country,
	count(r.*) as n_relations
FROM
	relationships r
	LEFT OUTER JOIN organizations o ON (o.id = r.id1)
WHERE
	o.status = 'approved' AND r.status = 'suggested' AND (r.reltype = 'IsChildOf' OR r.reltype = 'IsParentOf')
GROUP BY o.id, o.name, o.city, o.country
ORDER BY o.name;

CREATE VIEW persistent_orgs_view AS SELECT
	po.id,
	substr(po.id, 1, 14)||md5(substr(po.id,15)) as openaire_id,
	true as openaire_persistent,
	o.name,
	o.city,
	o.country
FROM
	persistent_orgs po
	JOIN organizations o ON (po.id = o.id)
ORDER BY o.name;

CREATE VIEW api_journal_view AS SELECT
	o.id          AS org_id,
	o.name        AS org_name,
	o.country     AS country,
	j.operation   AS operation,
	j.description AS description,
	j.op_date     AS op_date
FROM organizations o JOIN journal j ON o.id = j.id;

COMMIT;
