BEGIN;
DROP VIEW IF EXISTS organizations_simple_view;

DROP TRIGGER IF EXISTS insert_or_update_index_search_trigger ON journal;
DROP TRIGGER IF EXISTS insert_or_update_index_search_trigger ON organizations;
DROP TRIGGER IF EXISTS delete_index_search_trigger ON organizations;

DROP FUNCTION IF EXISTS refresh_index_search();
DROP FUNCTION IF EXISTS delete_index_search();
DROP FUNCTION IF EXISTS insert_or_update_index_search_trigger();
DROP FUNCTION IF EXISTS fulltext_search (tsvector, text);

DROP TABLE org_index_search;

CREATE TABLE org_index_search(
	id                    text PRIMARY KEY,
	txt                   tsvector,
	openaire_id           text,
	name                  text, 
	type                  text, 
	city                  text, 
	country               text, 
	status                text, 
	acronyms              text[], 
	othernames            text[], 
	urls                  text[], 
	pids                  text[], 
	n_similar_dups        bigint, 
	n_suggested_dups      bigint, 
	n_different_dups      bigint,
	name_norm             text, 
	type_norm             text, 
	city_norm             text, 
	country_norm          text, 
	status_norm           text, 
	acronyms_norm         text, 
	othernames_norm       text, 
	urls_norm             text, 
	pids_norm             text
);
CREATE INDEX org_index_search_txt_gin_idx ON org_index_search USING gin(txt);
CREATE INDEX org_index_search_openaire_id_idx ON org_index_search(openaire_id);
CREATE INDEX org_index_search_name_norm_idx ON org_index_search(name_norm);
CREATE INDEX org_index_search_type_norm_idx ON org_index_search(type_norm); 
CREATE INDEX org_index_search_city_norm_idx ON org_index_search(city_norm);
CREATE INDEX org_index_search_country_norm_idx ON org_index_search(country_norm); 
CREATE INDEX org_index_search_status_norm_idx ON org_index_search(status_norm);

CREATE OR REPLACE FUNCTION fulltext_search(vector tsvector, s text) RETURNS boolean AS $$
        BEGIN
                RETURN vector @@ plainto_tsquery(s);
        END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION refresh_index_search() RETURNS bigint AS $$
		DELETE FROM org_index_search;
		WITH t as (
			INSERT INTO org_index_search(
				id,
				txt,
				openaire_id,
				name, 
				type, 
				city, 
				country, 
				status, 
				acronyms, 
				othernames, 
				urls, 
				pids, 
				n_similar_dups, 
				n_suggested_dups, 
				n_different_dups,
				name_norm, 
				type_norm, 
				city_norm, 
				country_norm, 
				status_norm, 
				acronyms_norm, 
				othernames_norm, 
				urls_norm, 
				pids_norm
			) SELECT
				o.id, 
				to_tsvector(o.id||' '||substr(o.id, 1, 14)||md5(substr(o.id, 15))||' '||o.name||' '||array_to_string(array_agg(DISTINCT n.name), ' ','')||' '||array_to_string(array_agg(DISTINCT i.otherid), ' ','')||' '||array_to_string(array_agg(DISTINCT a.acronym), ' ','')||' '||array_to_string(array_agg(DISTINCT u.url), ' ','')),
				substr(o.id, 1, 14)||md5(substr(o.id, 15)),
				o.name,
				o.type,
				o.city,
				o.country,
				o.status,
				array_remove(array_agg(DISTINCT a.acronym), NULL),
				array_remove(array_agg(DISTINCT n.name), NULL),
				array_remove(array_agg(DISTINCT u.url), NULL),
				array_remove(array_agg(DISTINCT i.otherid), NULL),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_similar'  ),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'suggested'   ),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_different'),
				lower(trim(o.name)), 
				lower(trim(o.type)), 
				lower(trim(o.city)), 
				lower(trim(o.country)), 
				lower(trim(o.status)),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(a.acronym)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(n.name)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(u.url)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(i.otherid)), NULL), '§')||'§')
			FROM organizations o 
				LEFT OUTER JOIN other_names n on (o.id = n.id)
				LEFT OUTER JOIN other_ids i on (o.id = i.id)
				LEFT OUTER JOIN acronyms a on (o.id = a.id)
				LEFT OUTER JOIN urls u on (o.id = u.id)
				LEFT OUTER JOIN oa_duplicates d1 ON (o.id = d1.local_id)
			GROUP BY o.id, o.name, o.type, o.city, o. country, o.status RETURNING *
		) SELECT COUNT(*) FROM t;
$$ LANGUAGE SQL;

SELECT refresh_index_search();

CREATE OR REPLACE FUNCTION delete_index_search() RETURNS trigger LANGUAGE plpgsql AS $$
BEGIN
	DELETE FROM org_index_search WHERE id = old.id;
	RETURN OLD;
END;
$$;

CREATE OR REPLACE FUNCTION insert_or_update_index_search_trigger() RETURNS trigger LANGUAGE plpgsql AS $$
BEGIN
	INSERT INTO org_index_search(
		id,
		txt,
		openaire_id,
		name, 
		type, 
		city, 
		country, 
		status, 
		acronyms, 
		othernames, 
		urls, 
		pids, 
		n_similar_dups, 
		n_suggested_dups, 
		n_different_dups,
		name_norm, 
		type_norm, 
		city_norm, 
		country_norm, 
		status_norm, 
		acronyms_norm, 
		othernames_norm, 
		urls_norm, 
		pids_norm
	) (
		SELECT
			o.id, 
			to_tsvector(o.id||' '||substr(o.id, 1, 14)||md5(substr(o.id, 15))||' '||o.name||' '||array_to_string(array_agg(DISTINCT n.name), ' ','')||' '||array_to_string(array_agg(DISTINCT i.otherid), ' ','')||' '||array_to_string(array_agg(DISTINCT a.acronym), ' ','')||' '||array_to_string(array_agg(DISTINCT u.url), ' ','')),
			substr(o.id, 1, 14)||md5(substr(o.id, 15)),
			o.name,
			o.type,
			o.city,
			o.country,
			o.status,
			array_remove(array_agg(DISTINCT a.acronym), NULL),
			array_remove(array_agg(DISTINCT n.name), NULL),
			array_remove(array_agg(DISTINCT u.url), NULL),
			array_remove(array_agg(DISTINCT i.otherid), NULL),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_similar'  ),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'suggested'   ),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_different'),
			lower(trim(o.name)), 
			lower(trim(o.type)), 
			lower(trim(o.city)), 
			lower(trim(o.country)), 
			lower(trim(o.status)),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(a.acronym)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(n.name)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(u.url)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(i.otherid)), NULL), '§')||'§')
		FROM organizations o 
		LEFT OUTER JOIN other_names n on (o.id = n.id)
		LEFT OUTER JOIN other_ids i on (o.id = i.id)
		LEFT OUTER JOIN acronyms a on (o.id = a.id)
		LEFT OUTER JOIN urls u on (o.id = u.id)
		LEFT OUTER JOIN oa_duplicates d1 ON (o.id = d1.local_id)
		WHERE o.id = new.id
		GROUP BY o.id, o.name, o.type, o.city, o. country, o.status
	) ON CONFLICT (id) DO UPDATE SET (
		txt,
		openaire_id,
		name, 
		type, 
		city, 
		country, 
		status, 
		acronyms, 
		othernames, 
		urls, 
		pids, 
		n_similar_dups, 
		n_suggested_dups, 
		n_different_dups,
		name_norm, 
		type_norm, 
		city_norm, 
		country_norm, 
		status_norm, 
		acronyms_norm, 
		othernames_norm, 
		urls_norm, 
		pids_norm
	) = (
		EXCLUDED.txt,
		EXCLUDED.openaire_id,
		EXCLUDED.name, 
		EXCLUDED.type, 
		EXCLUDED.city, 
		EXCLUDED.country, 
		EXCLUDED.status, 
		EXCLUDED.acronyms, 
		EXCLUDED.othernames, 
		EXCLUDED.urls, 
		EXCLUDED.pids, 
		EXCLUDED.n_similar_dups, 
		EXCLUDED.n_suggested_dups, 
		EXCLUDED.n_different_dups,
		EXCLUDED.name_norm, 
		EXCLUDED.type_norm, 
		EXCLUDED.city_norm, 
		EXCLUDED.country_norm, 
		EXCLUDED.status_norm, 
		EXCLUDED.acronyms_norm, 
		EXCLUDED.othernames_norm, 
		EXCLUDED.urls_norm, 
		EXCLUDED.pids_norm
	); 
	RETURN NEW;
END;
$$;

-- I create a trigger on the journal to manage all the insert/update operation (metadata, duplicates, ...) and because it should be executed at the end of all the necessary updates.
CREATE TRIGGER insert_or_update_index_search_trigger AFTER INSERT  ON journal FOR EACH ROW EXECUTE PROCEDURE insert_or_update_index_search_trigger();
CREATE TRIGGER delete_index_search_trigger           BEFORE DELETE ON organizations FOR EACH ROW EXECUTE PROCEDURE delete_index_search();

COMMIT;
