package eu.dnetlib.dhp;

import eu.dnetlib.dhp.solr.RecordImporter;
import eu.dnetlib.dhp.utils.ArgumentApplicationParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.spark.SparkConf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Objects;
import java.util.Optional;

public class RecordImporterApplication {

    private static final Logger log = LoggerFactory.getLogger(RecordImporterApplication.class);

    private static final String APPLICATION_JAR = "./openaire-solr-record-importer.jar";
    private static final String APPLICATION_TITLE = "OpenAIRE Solr record importer";
    private static final String DEFAULT_TASKS = "*";

    public static void main(final String[] args) throws IOException, ParseException {

        final ArgumentApplicationParser cmd = parseArguments(args);

        // TO AVOID EXCEPTIONS WITH MANDATORY FIELDS
        for (final String s : args) {
            if (s.equals("-h") || s.equals("--help")) {
                printHelpAndExit(cmd.getOptions());
            }
        }

        log.info("**** EXECUTING - {} ***", APPLICATION_TITLE);

        final Boolean isSparkSessionManaged = Optional
                .ofNullable(cmd.getOptionValue("isSparkSessionManaged"))
                .map(Boolean::valueOf)
                .orElse(Boolean.TRUE);
        log.info("isSparkSessionManaged: {}", isSparkSessionManaged);

        final String path = cmd.getOptionValue("path");
        log.info("path: {}", path);

        final String collection = cmd.getOptionValue("collection");
        log.info("collection: {}", collection);

        final String zkHost = cmd.getOptionValue("zkHost");
        log.info("zkHost: {}", zkHost);

        final String tasks = Optional
                .ofNullable(cmd.getOptionValue("tasks"))
                .orElse(DEFAULT_TASKS);
        log.info("tasks: {}", tasks);

        final int batchSize = Optional
                .ofNullable(cmd.getOptionValue("batchSize"))
                .map(Integer::parseInt)
                .orElse(RecordImporter.BATCH_SIZE);
        log.info("batchSize: {}", tasks);

        final Boolean shouldCommit = Optional
                .ofNullable(cmd.getOptionValue("shouldCommit"))
                .map(Boolean::valueOf)
                .orElse(Boolean.FALSE);
        log.info("shouldCommit: {}", shouldCommit);

        final Boolean shouldFilterXmlPayload = Optional
                .ofNullable(cmd.getOptionValue("shouldFilterXmlPayload"))
                .map(Boolean::valueOf)
                .orElse(Boolean.FALSE);
        log.info("shouldFilterXmlPayload: {}", shouldFilterXmlPayload);

        final SparkConf conf = new SparkConf();

        if (Boolean.FALSE.equals(isSparkSessionManaged)) {
            conf.setAppName(APPLICATION_TITLE);
            conf.setMaster(String.format("local[%s]", tasks));
            conf.set("spark.driver.host", "localhost");
            conf.set("spark.ui.enabled", "false");
        }

        RecordImporter.importRecords(conf, zkHost, collection, path, batchSize, shouldCommit, shouldFilterXmlPayload);

        log.info("**** DONE ***");
    }

    private static ArgumentApplicationParser parseArguments(String[] args) throws IOException, ParseException {
        return ArgumentApplicationParser.parse(
                IOUtils.toString(
                        Objects.requireNonNull(RecordImporterApplication.class.getResourceAsStream(
                                "/eu/dnetlib/dhp/input_parameters.json")), Charset.defaultCharset()), args);
    }

    private static void printHelpAndExit(final Options options) {
        final String ln = StringUtils.repeat("=", APPLICATION_TITLE.length());
        System.out.println(String.format("\n%s\n%s\n%s\n", ln, APPLICATION_TITLE, ln));
        new HelpFormatter().printHelp(APPLICATION_JAR, options, true);

        System.exit(1);
    }
}