package eu.dnetlib.dhp.solr;

import com.google.common.base.Splitter;
import com.google.common.cache.*;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

public class CacheCloudSolrClient implements Serializable {

    private static final Logger log = LoggerFactory.getLogger(CacheCloudSolrClient.class);

    private static final CacheLoader<CloudClientParams, CloudSolrClient> loader = new CacheLoader<>() {
        @Override
        public CloudSolrClient load(CloudClientParams params) throws Exception {
            return getCloudSolrClient(params);
        }
    };

    private static final RemovalListener<CloudClientParams, CloudSolrClient> listener = rn -> Optional
            .ofNullable(rn.getValue())
            .ifPresent(client -> {
                try {
                    client.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
            }});

    private static final LoadingCache<CloudClientParams, CloudSolrClient> cache = CacheBuilder
            .newBuilder()
            .removalListener(listener)
            .build(loader);

    public static void invalidateCachedClient(CloudClientParams params) {
        cache.invalidate(params);
    }

    public static CloudSolrClient getCachedCloudClient(CloudClientParams params) throws ExecutionException {
        return CacheCloudSolrClient.cache.get(params);
    }

    private static CloudSolrClient getCloudSolrClient(CloudClientParams cloudClientParams) {
        String zkHost = cloudClientParams.getZkHost();
        log.info("Creating a new SolrCloudClient for zkhost {}", zkHost);

        final List<String> zkUrlList = StreamSupport.stream(
                Splitter.on(",")
                        .split(zkHost).spliterator(), false)
                .collect(Collectors.toList());

        CloudSolrClient.Builder solrClientBuilder = new CloudSolrClient.Builder(zkUrlList, Optional.empty());

        // HttpSolrClient.Builder httpSolrClientBuilder = new HttpSolrClient.Builder();
        // http2SolrClientBuilder.connectionTimeout(1, TimeUnit.MINUTES);
        // http2SolrClientBuilder.idleTimeout(2);
        // solrClientBuilder.withHttpClient(httpSolrClientBuilder.build().getHttpClient());

        CloudSolrClient solrClient = solrClientBuilder.build();
        solrClient.connect();
        log.debug("Created new SolrCloudClient for zkhost {}", zkHost);
        return solrClient;
    }

}
