
package eu.dnetlib.dhp.utils;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.cli.*;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.Charset;
import java.util.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

public class ArgumentApplicationParser implements Serializable {

	private static final Logger log = LoggerFactory.getLogger(ArgumentApplicationParser.class);

	private final Options options = new Options();
	private final Map<String, String> objectMap = new HashMap<>();

	private final List<String> compressedValues = new ArrayList<>();

	public static ArgumentApplicationParser parse(String jsonSpec, String[] args) throws IOException, ParseException {
		ArgumentApplicationParser parser = new ArgumentApplicationParser(jsonSpec);
		return parser.parseArgument(args);
	}

	public ArgumentApplicationParser(final String jsonConfiguration) throws IOException {
		final ObjectMapper mapper = new ObjectMapper();
		final OptionsParameter[] configuration = mapper.readValue(jsonConfiguration, OptionsParameter[].class);
		createOptionMap(configuration);
	}

	private void createOptionMap(final OptionsParameter[] configuration) {
		Arrays
			.stream(configuration)
			.map(
				conf -> {
					final Option o = new Option(conf.getParamName(), true, conf.getParamDescription());
					o.setLongOpt(conf.getParamLongName());
					o.setRequired(conf.isParamRequired());
					if (conf.isCompressed()) {
						compressedValues.add(conf.getParamLongName());
					}
					return o;
				})
			.forEach(options::addOption);
	}

	private static String decompressValue(final String abstractCompressed) {
		try {
			byte[] byteArray = Base64.decodeBase64(abstractCompressed.getBytes());
			GZIPInputStream gis = new GZIPInputStream(new ByteArrayInputStream(byteArray));
			final StringWriter stringWriter = new StringWriter();
			IOUtils.copy(gis, stringWriter, Charset.defaultCharset());
			return stringWriter.toString();
		} catch (IOException e) {
			log.error("Wrong value to decompress: {}", abstractCompressed);
			throw new IllegalArgumentException(e);
		}
	}

	public ArgumentApplicationParser parseArgument(final String[] args) throws ParseException {
		CommandLineParser parser = new DefaultParser();
		CommandLine cmd = parser.parse(options, args);
		Arrays
			.stream(cmd.getOptions())
			.forEach(
				it -> objectMap
					.put(
						it.getLongOpt(),
						compressedValues.contains(it.getLongOpt())
							? decompressValue(it.getValue())
							: it.getValue()));
		return this;
	}

	public boolean hasOption(final String opt) {
		return options.hasOption(opt);
	}

	public Options getOptions() {
		return options;
	}

	public String getOptionValue(final String key) {
		return objectMap.get(key);
	}

	public Map<String, String> getObjectMap() {
		return objectMap;
	}
}
