package eu.dnetlib.dhp.solr;

import com.google.common.base.Splitter;
import com.google.common.cache.*;

import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.client.config.RequestConfig;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.StreamSupport;

public class CacheCloudSolrClient implements Serializable {

    private static final Logger log = LoggerFactory.getLogger(CacheCloudSolrClient.class);

    private static final CacheLoader<CloudClientParams, CloudSolrClient> loader = new CacheLoader<>() {
        @Override
        public CloudSolrClient load(CloudClientParams params) throws Exception {
            return getCloudSolrClient(params);
        }
    };

    private static final RemovalListener<CloudClientParams, CloudSolrClient> listener = rn -> Optional
            .ofNullable(rn.getValue())
            .ifPresent(client -> {
                try {
                    client.close();
                } catch (IOException e) {
                    throw new RuntimeException(e);
            }});

    private static final LoadingCache<CloudClientParams, CloudSolrClient> cache = CacheBuilder
            .newBuilder()
            .removalListener(listener)
            .build(loader);

    public static void invalidateCachedClient(CloudClientParams params) {
        cache.invalidate(params);
    }

    public static CloudSolrClient getCachedCloudClient(CloudClientParams params) throws ExecutionException {
        return CacheCloudSolrClient.cache.get(params);
    }

    private static CloudSolrClient getCloudSolrClient(CloudClientParams cloudClientParams) {
        String zkHost = cloudClientParams.getZkHost();
        log.info("Creating a new SolrCloudClient for zkhost {}", zkHost);

        final List<String> zkUrlList = StreamSupport.stream(
                        Splitter.on(",")
                                .split(zkHost).spliterator(), false)
                .toList();

        // Configure Apache HttpClient
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(cloudClientParams.getHttpConnectTimeoutMillis())
                .setSocketTimeout(cloudClientParams.getHttpSocketTimeoutMillis())
                .build();

        CloseableHttpClient httpClient = HttpClientBuilder.create()
                .setMaxConnTotal(cloudClientParams.getMaxConnTotal()) // Maximum total connections
                .setMaxConnPerRoute(cloudClientParams.getMaxConnPerRoute()) // Maximum connections per host
                .setDefaultRequestConfig(requestConfig)
                .build();

        // Configure CloudSolrClient.Builder
        CloudSolrClient.Builder cloudSolrClientBuilder = new CloudSolrClient.Builder(zkUrlList, Optional.empty())
                .withHttpClient(httpClient);

        CloudSolrClient solrClient = cloudSolrClientBuilder.build();
        solrClient.setZkClientTimeout(cloudClientParams.getZkClientTimeout());
        solrClient.setZkConnectTimeout(cloudClientParams.getZkConnectTimeout());
        solrClient.connect();
        log.debug("Created new SolrCloudClient for zkhost {}", zkHost);
        return solrClient;
    }

}
