package eu.dnetlib.dhp.solr.mapping;

import org.apache.commons.lang3.StringUtils;
import org.apache.solr.common.SolrInputDocument;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class MappingUtils implements Serializable {

    public static final String BLANK = "";

    public static final String INFO_EU_REPO_DATE_PUBLICATION = "info:eu-repo/date/publication/";

    private static final String NORMALIZE_OUT_FORMAT = "yyyy-MM-dd'T'hh:mm:ss'Z'";

    private static final String[] NORMALIZE_INPUT_FORMATS = {
            NORMALIZE_OUT_FORMAT, "yyyy-MM-dd'T'hh:mm:ss", "yyyy/MM/dd", "yyyy-MM-dd", "yyyy"
    };

    private static final String[] dateFormats = {
            NORMALIZE_OUT_FORMAT, "yyyy/MM/dd"
    };

    public static String extractYear(String s) {
        Calendar c = new GregorianCalendar();
        for (String format : dateFormats) {
            try {
                c.setTime(new SimpleDateFormat(format).parse(s));
                return String.valueOf(c.get(Calendar.YEAR));
            } catch (ParseException e) {
            }
        }
        return "";
    }

    public static String normalizeDate(String s) {
        final String date = StringUtils.isNotBlank(s) ? s.replace(INFO_EU_REPO_DATE_PUBLICATION, BLANK).trim() : BLANK;

        for (String format : NORMALIZE_INPUT_FORMATS) {
            try {
                Date parse = new SimpleDateFormat(format).parse(date);
                return new SimpleDateFormat(NORMALIZE_OUT_FORMAT).format(parse);
            } catch (ParseException e) {
            }
        }
        return BLANK;
    }

    public static void addField(String fieldName, Optional<String> value, SolrInputDocument doc) {
        value.ifPresent(val -> addField(fieldName, Collections.singletonList(val), new ArrayList<>(), doc));
    }

    public static void addField(String fieldName, String value, SolrInputDocument doc) {
        addField(fieldName, Collections.singletonList(value), new ArrayList<>(), doc);
    }

    public static void addField(String fieldName, List<String> value, SolrInputDocument doc) {
        Optional.ofNullable(value)
                .ifPresent(val -> addField(fieldName, val, new ArrayList<>(), doc));
    }

    public static void addField(String fieldName, String value, List<String> otherVals, SolrInputDocument doc) {
        addField(fieldName, Collections.singletonList(value), otherVals, doc);
    }

    public static void addField(String fieldName, List<String> value, List<String> otherVals, SolrInputDocument doc) {
        final Set<String> values = Stream
                .concat(
                        Optional.ofNullable(value).orElse(new ArrayList<>()).stream(),
                        Optional.ofNullable(otherVals).orElse(new ArrayList<>()).stream())
                .collect(Collectors.toCollection(HashSet::new));
        if (!values.isEmpty()) {
            doc.addField(fieldName, values);
        }
    }

}
