package eu.dnetlib.data.collector.plugins.czechProject;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.XML;

import eu.dnetlib.data.collector.plugin.AbstractCollectorPlugin;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.InterfaceDescriptor;

public class CzechProjectsPlugin extends AbstractCollectorPlugin {

	private static final Log log = LogFactory.getLog(CzechProjectsPlugin.class);

	@Override
	public Iterable<String> collect(final InterfaceDescriptor interfaceDescriptor, final String fromDate, final String untilDate)
		throws CollectorServiceException {

		final String baseUrl = interfaceDescriptor.getBaseUrl();
		final String token = interfaceDescriptor.getParams().get("token");
		final int fromYear = NumberUtils.toInt(interfaceDescriptor.getParams().get("fromYear"), 0);
		final int toYear = NumberUtils.toInt(interfaceDescriptor.getParams().get("toYear"), 0);

		if (StringUtils.isBlank(baseUrl)) { throw new CollectorServiceException("baseUrl is empty"); }

		if (StringUtils.isBlank(token)) { throw new CollectorServiceException("Authorization token is empty"); }

		if (fromYear < 1900 || toYear < 1900 || toYear < fromYear) { throw new CollectorServiceException("Invalid year range"); }

		return () -> IntStream.rangeClosed(fromYear, toYear).mapToObj(year -> fetchProjects(baseUrl, token, year)).flatMap(List::stream).iterator();
	}

	private List<String> fetchProjects(final String url, final String token, final int year) {

		try (final CloseableHttpClient client = HttpClients.createDefault()) {

			final List<NameValuePair> urlParameters = new ArrayList<>();
			urlParameters.add(new BasicNameValuePair("token", token));
			urlParameters.add(new BasicNameValuePair("rezim", "filtr-detail"));
			urlParameters.add(new BasicNameValuePair("oblast", "cep"));
			urlParameters.add(new BasicNameValuePair("sp-rok-zahajeni-od", Integer.toString(year)));
			urlParameters.add(new BasicNameValuePair("sp-rok-zahajeni-do", Integer.toString(year)));

			final HttpPost req = new HttpPost(url);
			req.addHeader("Accept", "application/json");
			req.setEntity(new UrlEncodedFormEntity(urlParameters));

			try (final CloseableHttpResponse response = client.execute(req)) {
				return toXmlList(response.getEntity().getContent());
			}
		} catch (final IOException e) {
			log.error("Error fetching projects", e);
			throw new RuntimeException("Error fetching projects", e);
		}

	}

	protected List<String> toXmlList(final InputStream is) throws IOException {
		final String content = IOUtils.toString(is);
		final JSONObject obj = new JSONObject(content);
		final JSONArray jsonList = obj.getJSONArray("data");
		return StreamSupport.stream(jsonList.spliterator(), false).map(r -> XML.toString(r, "record")).collect(Collectors.toList());
	}
}
