package eu.dnetlib.data.collector.plugins.doiresolver;

import eu.dnetlib.data.collector.plugins.filesystem.FileSystemIterator;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

public class DOIResolverIterator implements Iterator<String> {

    private static final Log log = LogFactory.getLog(DOIResolverIterator.class);

    protected static final String STARTER = "FIRE";
    protected static final String TERMINATOR = "ARNOLD";
    protected static final String BAD_TERMINATOR = "BAD";
    protected static final String UNRESOLVED = "UNRESOLVED";
    protected static long TIMEOUT = 5;
    protected static TimeUnit TIMEOUT_UNIT = TimeUnit.SECONDS;

    /** Path to the dir that contains the files, each a csv with a list of DOIs, one per line. **/
    private String baseDir;
    private String fromDate;

    private ArrayBlockingQueue<String> queue;

    private CrossrefResolver crossrefResolver;


    public DOIResolverIterator(final String baseDir, final CrossrefResolver crossrefResolver, final String fromDate) {
        this.baseDir = baseDir;
        this.fromDate = fromDate;
        this.queue = new ArrayBlockingQueue<>(100);
        this.crossrefResolver = crossrefResolver;
        init();
    }

    private void init(){
        log.info("Init");

        new Thread(() -> {
            try{
                final FileSystemIterator fsi = new FileSystemIterator(baseDir, "csv", fromDate);
                // put first item in the queue
                if(queue.offer(STARTER)) {
                    // read the file, ask the resolvers, put results in a shared queue
                    //whatever exceptions, add terminator to the queue
                    while (fsi.hasNext()) {
                        String filePath = fsi.next();
                        try (Stream<String> stream = Files.lines(Paths.get(filePath))) {

                            stream.forEach(doi -> {
                                try {
                                    String resolved = resolve(doi);
                                    if(!resolved.equals(UNRESOLVED)) queue.offer(resolved, TIMEOUT, TIMEOUT_UNIT);
                                } catch (InterruptedException e) {
                                    log.error("DOI processing aborted, cannot offer resolved doi: "+doi+" . Did the consumer die?");
                                    log.error(e);
                                    queue.offer(BAD_TERMINATOR);
                                }
                            });

                        } catch (IOException e) {
                            log.error("DOI processing aborted");
                            log.error(e);
                            queue.offer(BAD_TERMINATOR);
                        }
                    }
                }
            } catch (Exception e) {
                log.error("DOI processing aborted");
                log.error(e);
                queue.offer(BAD_TERMINATOR);
            }
            queue.offer(TERMINATOR);
            log.info("Finished processing DOI list");
        }
        ).start();
    }

    private String resolve(final String doi){
       log.debug("Resolving "+doi);
       log.debug("Crossref...");
       String record = crossrefResolver.resolve(cleanDOI(doi));
       if(StringUtils.isNotBlank(record)) return record;
       else {
           //try another resolver
           log.debug("Resolver returned blank item");
       }
       return UNRESOLVED;
    }

    /**
     * Returns the identifier part of the DOI only.
     * @param doi
     * @return the DOI
     */
    protected String cleanDOI(final String doi){
       return doi.replace("http://dx.doi.org/", "").replace("https://dx.doi.org/", "")
               .replace("https://doi.org/", "").replace("http://doi.org/", "");
    }

    @Override
    public boolean hasNext() {
       return doHasNext();
    }

    private boolean doHasNext(){
        //If I get a null value, the queue is currently empty. so we wait for something
        String element = queue.peek();
        while(element == null) {
            try {
                log.debug("Sleeping while waiting for something in the queue");
                Thread.sleep(1000);
                element = queue.peek();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        log.debug("Found in queue element: "+element);
        switch(element){
            case TERMINATOR:
            case BAD_TERMINATOR:
                return false;
            case STARTER:
            case UNRESOLVED: //although they should not be inserted at all in the queue
                queue.poll();
                return doHasNext();
            default:
                return true;
        }
    }

    @Override
    public String next() {
        return queue.poll();
    }

    public String getBaseDir() {
        return baseDir;
    }

    public void setBaseDir(String baseDir) {
        this.baseDir = baseDir;
    }

    public CrossrefResolver getCrossrefResolver() {
        return crossrefResolver;
    }

    public void setCrossrefResolver(CrossrefResolver crossrefResolver) {
        this.crossrefResolver = crossrefResolver;
    }
}
