package eu.dnetlib.data.collector.plugins.doiresolver;

import eu.dnetlib.data.collector.plugins.HttpConnector;
import eu.dnetlib.data.collector.plugins.utils.JsonUtils;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.InputStream;
import java.util.regex.Pattern;

public class CrossrefResolver implements DOIResolver{

    private static final Log log = LogFactory.getLog(CrossrefResolver.class);
    private String baseURL = "https://api.crossref.org/works/";

    public static Pattern CROSSREF_DOI_REGEXP  = Pattern.compile("^10\\.\\d{4,9}/[-._;()/:A-Z0-9]+$", Pattern.CASE_INSENSITIVE);


    @Autowired
    private HttpConnector httpConnector;

    @Override
    public String resolve(String doi) {
        if(isValid(doi)) {
            try {
                InputStream is = httpConnector.getInputSourceAsStream(getBaseURL() + doi);
                return asXml(IOUtils.toString(is));
            } catch (IOException | CollectorServiceException e) {
                log.error("Cannot resolve doi " + doi + " Exception: " + e);
                return null;
            }
        }
        else {
            log.debug("Input does not look like a valid DOI " + doi);
            return null;
        }
    }

    protected boolean isValid(String doi){
        return CROSSREF_DOI_REGEXP.matcher(doi).matches();
    }

    protected String asXml(final String record){
       return new JsonUtils().convertToXML(record);
    }

    public String getBaseURL() {
        return baseURL;
    }

    public void setBaseURL(final String baseURL) {
        this.baseURL = baseURL;
    }

    public HttpConnector getHttpConnector() {
        return httpConnector;
    }

    public void setHttpConnector(HttpConnector httpConnector) {
        this.httpConnector = httpConnector;
    }
}
