package eu.dnetlib.data.collector.plugins.gtr2;

import java.util.Iterator;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;

/**
 * Plugin to collect metadata record about projects and fundings via the UKRI grt2 API.
 * <p>
 * Documentation : http://gtr.ukri.org/resources/api.html.
 * </p>
 * <p>
 * BaseURL: https://gtr.ukri.org/gtr/api The results returned by the API are XMLs.
 * </p>
 * <p>
 * Pagination: TO BE DEFINED. Exceeding the number of pages available will result in a HTTP response code of 404
 * </p>
 *
 * @author alessia
 */
public class Gtr2ProjectsCollectorPlugin extends AbstractGtr2CollectorPlugin {

	@Override
	protected Iterator<String> createIterator(final String baseUrl, final String fromDate, final String startPage, final String endPage)
		throws CollectorServiceException {

		return new Gtr2Iterator(baseUrl, fromDate, startPage, endPage) {

			@Override
			protected String urlForPage(final String baseUrl, final int pageNumber) {
				return baseUrl + "/projects?p=" + pageNumber;
			}

			@Override
			protected String xpathForEntity() {
				return "//*[local-name() = 'project']";
			}

			@Override
			protected String expandMainEntity(final Element mainEntity) {

				final Element newRoot = DocumentHelper.createElement("doc");

				newRoot.add(mainEntity);

				addLinkedEntities(mainEntity, "LEAD_ORG", newRoot, o -> asOrgElement("ld-org", o));
				addLinkedEntities(mainEntity, "PP_ORG", newRoot, o -> asOrgElement("pp-org", o));
				addLinkedEntities(mainEntity, "PI_PER", newRoot, o -> asPersonElement("pi-per", o));
				addLinkedEntities(mainEntity, "FUND", newRoot, o -> asFundElement("funding", o));

				return DocumentHelper.createDocument(newRoot).asXML();
			}

			private Element asOrgElement(final String nodeName, final Document doc) {
				final Element newOrg = DocumentHelper.createElement(nodeName);
				newOrg.addElement("id").setText(doc.valueOf("/*/@*[local-name()='id']"));
				newOrg.addElement("name").setText(doc.valueOf("//*[local-name()='name']"));
				newOrg.addElement("country").setText(doc.valueOf("//*[local-name()='country']"));
				return newOrg;
			}

			private Element asPersonElement(final String nodeName, final Document doc) {
				final Element newPers = DocumentHelper.createElement(nodeName);
				newPers.addElement("id").setText(doc.valueOf("/*/@*[local-name()='id']"));
				newPers.addElement("firstName").setText(doc.valueOf("//*[local-name()='firstName']"));
				newPers.addElement("otherNames").setText(doc.valueOf("//*[local-name()='otherNames']"));
				newPers.addElement("surname").setText(doc.valueOf("//*[local-name()='surname']"));
				return newPers;
			}

			private Element asFundElement(final String nodeName, final Document doc) {
				final Element newFund = DocumentHelper.createElement(nodeName);
				newFund.addElement("id").setText(doc.valueOf("/*/@*[local-name()='id']"));
				newFund.addElement("start").setText(doc.valueOf("//*[local-name()='start']"));
				newFund.addElement("end").setText(doc.valueOf("//*[local-name()='end']"));
				newFund.addElement("valuePoundsAmount").setText(doc.valueOf("//*[local-name()='valuePounds']/@*[local-name()='amount']"));
				newFund.addElement("valuePoundsCurrencyCode").setText(doc.valueOf("//*[local-name()='valuePounds']/@*[local-name()='currencyCode']"));
				newFund.addElement("category").setText(doc.valueOf("//*[local-name()='category']"));
				return newFund;
			}
		};
	}

}
