package eu.dnetlib.msro.oai.workflows.nodes;

import javax.annotation.Resource;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.data.information.oai.utils.OAIParameterNames;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * The Class PrepareOaiDataJobNode. Sets the OAI parameters in the env using the constants in OAIParameterNames, to be used by subsequent nodes.
 */
public class PrepareOaiDataJobNode extends SimpleJobNode {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(PrepareOaiDataJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	/** The service locator. */
	@Resource
	private UniqueServiceLocator serviceLocator;

	private String oaiDbName;
	private String oaiFormat;
	private String oaiLayout;
	private String oaiInterpretation;
	private String oaiSource;

	private boolean skipDuplicates = false;
	private String duplicateXPath;

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.msro.workflows.nodes.SimpleJobNode#execute(com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected String execute(final NodeToken token) throws Exception {

		log.info("start preparing job");
		token.getEnv().setAttribute(OAIParameterNames.OAI_ACTION_DATE, DateUtils.now_ISO8601());
		setIfNotBlank(OAIParameterNames.OAI_DB, oaiDbName, token);
		setIfNotBlank(OAIParameterNames.OAI_FORMAT_NAME, oaiFormat, token);
		setIfNotBlank(OAIParameterNames.OAI_FORMAT_LAYOUT, oaiLayout, token);
		setIfNotBlank(OAIParameterNames.OAI_FORMAT_INTERPRETATION, oaiInterpretation, token);
		setIfNotBlank(OAIParameterNames.OAI_COLLECTON, oaiFormat + "-" + oaiLayout + "-" + oaiInterpretation, token);

		String configurationProfile = getConfigurationProfile();
		setIfNotBlank(OAIParameterNames.OAI_CONFIGURATION_PROFILE, configurationProfile, token);
		token.getEnv().setAttribute(OAIParameterNames.OAI_SKIP_DUPLICATES, String.valueOf(skipDuplicates));
		if(skipDuplicates) token.getEnv().setAttribute(OAIParameterNames.OAI_DUPLICATE_XPATH, duplicateXPath);
		return Arc.DEFAULT_ARC;
	}

	private void setIfNotBlank(final String attrName, final String attrValue, final NodeToken token){
		if(StringUtils.isNotBlank(attrValue))
			token.getEnv().setAttribute(attrName, attrValue);
	}

	/**
	 * Gets the configuration profile.
	 * 
	 * @return the configuration profile
	 * @throws MSROException
	 *             the MSRO exception
	 */
	private String getConfigurationProfile() throws MSROException {
		try {
			return this.serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
					"//RESOURCE_PROFILE[.//RESOURCE_TYPE/@value = 'OAIPublisherConfigurationDSResourceType']");
		} catch (Exception e) {
			throw new MSROException("Cannot find OAI configuration profile", e);
		}
	}

	public String getOaiDbName() {
		return oaiDbName;
	}

	public void setOaiDbName(final String oaiDbName) {
		this.oaiDbName = oaiDbName;
	}

	public String getOaiFormat() {
		return oaiFormat;
	}

	public void setOaiFormat(final String oaiFormat) {
		this.oaiFormat = oaiFormat;
	}

	public String getOaiLayout() {
		return oaiLayout;
	}

	public void setOaiLayout(final String oaiLayout) {
		this.oaiLayout = oaiLayout;
	}

	public String getOaiInterpretation() {
		return oaiInterpretation;
	}

	public void setOaiInterpretation(final String oaiInterpretation) {
		this.oaiInterpretation = oaiInterpretation;
	}

	public String getOaiSource() {
		return oaiSource;
	}

	public void setOaiSource(final String oaiSource) {
		this.oaiSource = oaiSource;
	}

	public boolean isSkipDuplicates() {
		return skipDuplicates;
	}

	public void setSkipDuplicates(final boolean skipDuplicates) {
		this.skipDuplicates = skipDuplicates;
	}

	public String getDuplicateXPath() {
		return duplicateXPath;
	}

	public void setDuplicateXPath(final String duplicateXPath) {
		this.duplicateXPath = duplicateXPath;
	}



}
