package eu.dnetlib.data.transform;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message.Builder;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import eu.dnetlib.data.proto.FieldTypeProtos.StringField;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.SpecialTrustProtos.SpecialTrust;
import eu.dnetlib.pace.config.DedupConfig;
import org.apache.commons.lang.StringUtils;

public class OafEntityMerger {

	private static final String DEDUP_CLASSID = "sysimport:dedup";

	private static final String DNET_PROVENANCE_SCHEME = "dnet:provenanceActions";

	private final Predicate<StringField> skipEmptyStringField = s -> (s != null) && (s.getValue() != null) && !s.getValue().isEmpty();

	private final Predicate<String> skipEmptyString = s -> StringUtils.isNotBlank(s);

	public static Oaf.Builder merge(final String id, final Iterable<Oaf> entities) {
		return merge(null, id, entities);
	}

	public static Oaf.Builder merge(final DedupConfig dedupConf, final String id, final Iterable<Oaf> entities) {
		return new OafEntityMerger().mergeEntities(dedupConf, id, entities);
	}

	public static Oaf.Builder merge(final Oaf.Builder builder) {
		return new OafEntityMerger().doMergeEntities(builder);
	}

	public Oaf.Builder mergeEntities(final DedupConfig dedupConf, final String id, final Iterable<Oaf> entities) {

		Oaf.Builder builder = Oaf.newBuilder();
		String trust = "0.0";



		for (final Oaf oaf : TrustOrdering.sort(entities)) {
			// doublecheck we're dealing only with main entities
			if (!oaf.getKind().equals(Kind.entity)) throw new IllegalArgumentException("expected OafEntity!");

			final String currentTrust = oaf.getDataInfo().getTrust();
			if (!currentTrust.equals(SpecialTrust.NEUTRAL.toString())) {
				trust = currentTrust;
			}
			builder.mergeFrom(oaf);
		}

		builder = doMergeEntities(builder);
		builder.getEntityBuilder().setId(id);
		builder.getDataInfoBuilder()
				.setInferred(true)
				.setDeletedbyinference(false)
				.setTrust(trust)
				.setInferenceprovenance(dedupConf != null ? dedupConf.getWf().getConfigurationId() : "")
				.setProvenanceaction(getProvenanceAction());

		if ((dedupConf != null) && dedupConf.getWf().isIncludeChildren()) {
			for (final Oaf oaf : Iterables.limit(entities, dedupConf.getWf().getMaxChildren())) {
				builder.getEntityBuilder().addChildren(oaf.getEntity());
			}
		}

		return builder;
	}

	private Qualifier.Builder getProvenanceAction() {
		return Qualifier.newBuilder().setClassid(DEDUP_CLASSID).setClassname(DEDUP_CLASSID).setSchemeid(DNET_PROVENANCE_SCHEME)
				.setSchemename(DNET_PROVENANCE_SCHEME);
	}

	public Oaf.Builder doMergeEntities(final Oaf.Builder builder) {

		for (final String field : OafUtils.getFieldNames(OafEntity.getDescriptor(), OafEntity.COLLECTEDFROM_FIELD_NUMBER)) {
			setKeyValues(builder.getEntityBuilder(), field);
		}
		for (final String field : OafUtils.getFieldNames(OafEntity.getDescriptor(), OafEntity.PID_FIELD_NUMBER)) {
			setStructuredProperty(builder.getEntityBuilder(), field);
		}
		for (final String field : OafUtils.getFieldNames(OafEntity.getDescriptor(), OafEntity.ORIGINALID_FIELD_NUMBER)) {
			setUniqueString(builder.getEntityBuilder(), field);
		}

		switch (builder.getEntity().getType()) {
		case datasource:
			break;
		case organization:
			break;
		case project:
			break;
		case result:
			final Result.Metadata.Builder result = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();
			setTitle(result);
			mergeInstances(builder.getEntityBuilder().getResultBuilder());

			// for (String field : Lists.newArrayList("subject", "relevantdate")) {
			for (final String field : OafUtils.getFieldNames(Result.Metadata.getDescriptor(), Result.Metadata.SUBJECT_FIELD_NUMBER,
					Result.Metadata.RELEVANTDATE_FIELD_NUMBER)) {
				setStructuredProperty(result, field);
			}
			for (final String field : OafUtils.getFieldNames(Result.Metadata.getDescriptor(), Result.Metadata.DESCRIPTION_FIELD_NUMBER)) {
				setLongestStringField(result, field);
			}
			for (final String field : OafUtils.getFieldNames(Result.Metadata.getDescriptor(), Result.Metadata.SOURCE_FIELD_NUMBER)) {
				setUniqueStringField(result, field);
			}

			// remove the inner authors, rely on the children
			builder.getEntityBuilder().getResultBuilder().getMetadataBuilder().clearAuthor();
			break;
		default:
			break;
		}
		return builder;
	}

	private void mergeInstances(final Result.Builder builder) {
		final Map<String, Instance.Builder> map = Maps.newHashMap();
		for(Instance i : builder.getInstanceList()) {

			final String key = i.getHostedby().getKey() + i.getAccessright().getClassid() + i.getInstancetype().getClassid();
			if (!map.containsKey(key)) {
				map.put(key, Instance.newBuilder().mergeFrom(i));
				continue;
			}

			map.get(key).addAllUrl(i.getUrlList());
		}

		for(Instance.Builder i : map.values()) {
			final Set<String> urls = Sets.newHashSet();
			urls.addAll(i.getUrlList());
			i.clearUrl().addAllUrl(urls);
		}
		builder.clearInstance();
		builder.addAllInstance(Iterables.transform(map.values(), b -> b.build()));
	}

	/**
	 * Helper method, avoid duplicated StructuredProperties in the given builder for the given fieldName
	 *
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setStructuredProperty(final Builder builder, final String fieldName) {
		final Map<String, StructuredProperty> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StructuredProperty> sps = (List<StructuredProperty>) builder.getField(fd);

		if ((sps != null) && !sps.isEmpty()) {
			for (final StructuredProperty sp : sps) {
				if (StringUtils.isNotBlank(sp.getValue())) {
					map.put(sp.getValue().toLowerCase(), sp);
				}
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	/**
	 * Helper method, avoid duplicated KeyValues in the given builder for the given fieldName
	 *
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setKeyValues(final Builder builder, final String fieldName) {
		final Map<String, KeyValue> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<KeyValue> kvs = (List<KeyValue>) builder.getField(fd);

		if ((kvs != null) && !kvs.isEmpty()) {
			for (final KeyValue sp : kvs) {
				map.put(sp.getKey(), sp);
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setSingleString(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);
		if ((field != null) && !field.isEmpty()) {
			final StringField s = (StringField) Iterables.getLast(Iterables.filter(field, skipEmptyStringField), "");

			if ((s != null) && (s.getValue() != null) && !s.getValue().isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(s));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setLongestStringField(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);

		if ((field != null) && !field.isEmpty()) {
			final StringField.Builder max = StringField.newBuilder().setValue("");
			int maxLength = 0;
			for (final StringField sf : field) {
				if (sf.getValue().length() > maxLength) {
					maxLength = sf.getValue().length();
					max.clear();
					max.mergeFrom(sf);
				}
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(max.build()));
		}
	}

	@SuppressWarnings("unchecked")
	private void setUniqueStringField(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StringField> field = (List<StringField>) builder.getField(fd);
		final Map<String, StringField> map = Maps.newHashMap();
		if ((field != null) && !field.isEmpty()) {
			for (final StringField s : Iterables.filter(field, skipEmptyStringField)) {
				map.put(s.getValue(), s);
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
		}
	}

	@SuppressWarnings("unchecked")
	private void setUniqueString(final Builder builder, final String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<String> field = (List<String>) builder.getField(fd);
		final Set<String> set = Sets.newHashSet();
		if ((field != null) && !field.isEmpty()) {
			for (final String s : Iterables.filter(field, skipEmptyString)) {
				set.add(s);
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(set));
		}
	}

	private void setTitle(final Result.Metadata.Builder metadata) {
		final Iterable<StructuredProperty> filtered = Iterables.filter(metadata.getTitleList(), OafUtils.mainTitleFilter());

		if (!Iterables.isEmpty(filtered)) {
			metadata.clearTitle().addTitle(Iterables.getLast(filtered));
		}
	}

}
