package eu.dnetlib.data.transform;

import java.util.List;

import eu.dnetlib.pace.model.FieldDef;
import org.apache.commons.lang.StringUtils;

import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;
import com.google.protobuf.Message;
import com.googlecode.protobuf.format.JsonFormat;

import eu.dnetlib.pace.config.Type;

/**
 * AbstractProtoMapper provide common navigation methods on the protocolbuffers Messages.
 *
 * @author claudio
 */
public abstract class AbstractProtoMapper {

	private static final String COND_WRAPPER = "\\{|\\}";
	private static final String COND_SEPARATOR = "#";
	/** The Constant PATH_SEPARATOR. */
	private static final String PATH_SEPARATOR = "/";

	/**
	 * Process multi path.
	 *
	 * @param proto
	 *            the proto
	 * @param paths
	 *            the paths
	 * @return the list
	 */
	protected List<Object> processMultiPath(final GeneratedMessage proto, final FieldDef fieldDef, final List<String> paths) {
		final List<Object> response = Lists.newArrayList();
		for (final String pathElements : paths) {
			response.addAll(processPath(proto, fieldDef, pathElements));
		}
		return response;
	}

	/**
	 * Process path.
	 *
	 * @param proto
	 *            the proto
	 * @param path
	 *            the path
	 * @return the list
	 */
	protected List<Object> processPath(final GeneratedMessage proto, final FieldDef fieldDef, final String path) {
		return processPath(proto, fieldDef, Lists.newLinkedList(Splitter.on(PATH_SEPARATOR).trimResults().split(path)));
	}

	/**
	 * Process path.
	 *
	 * @param proto
	 *            the proto
	 * @param pathElements
	 *            the list
	 * @return the list
	 */
	protected List<Object> processPath(final GeneratedMessage proto, final FieldDef fieldDef, final List<String> pathElements) {

		final List<Object> response = Lists.newArrayList();

		if (pathElements.isEmpty()) throw new RuntimeException("ProtoBuf navigation path is empty");

		final String fieldPathCond = pathElements.get(0);

		final String fieldPath = StringUtils.substringBefore(fieldPathCond, "[");
		final String cond = getCondition(fieldPathCond);

		final FieldDescriptor fd = proto.getDescriptorForType().findFieldByName(fieldPath);
		if ((fd != null)) {
			if (fd.isRepeated()) {
				final int count = proto.getRepeatedFieldCount(fd);
				for (int i = 0; i < count && count < fieldDef.getSize(); i++) {
					final Object field = proto.getRepeatedField(fd, i);
					response.addAll(generateFields(fd, field, fieldDef, pathElements, cond));
				}
			} else {
				final Object field = proto.getField(fd);
				response.addAll(generateFields(fd, field, fieldDef, pathElements, cond));
			}
		} else throw new IllegalArgumentException("Invalid protobuf path (field not found): " + StringUtils.join(pathElements, ">") + "\nMessage:\n" + proto);

		return response;
	}

	/**
	 * Generate fields.
	 *
	 * @param fd
	 *            the fd
	 * @param field
	 *            the field
	 * @param list
	 *            the list
	 * @return the list
	 */
	private List<Object> generateFields(final FieldDescriptor fd, final Object field, final FieldDef fieldDef, final List<String> list, final String cond) {

		final List<Object> res = Lists.newArrayList();
		if (field instanceof GeneratedMessage) {
			if (list.size() > 1) {

				if (StringUtils.isBlank(cond)) return processPath((GeneratedMessage) field, fieldDef, list.subList(1, list.size()));
				else {

					final List<String> condPath =
							Lists.newLinkedList(Splitter.on(COND_SEPARATOR).trimResults().split(StringUtils.substringBefore(cond, "=")));

					final String val = (String) Iterables.getOnlyElement(processPath((GeneratedMessage) field, fieldDef, condPath));
					final String condVal = StringUtils.substringAfter(cond, "=").replaceAll(COND_WRAPPER, "").trim();

					return val.equals(condVal) ? processPath((GeneratedMessage) field, fieldDef, list.subList(1, list.size())) : res;
				}
			}
			else if (Type.JSON.equals(fieldDef.getType())) {
				res.add(JsonFormat.printToString((Message) field));
				return res;
			} else throw new RuntimeException("No primitive type found");
		} else {
			if (list.size() == 1) {

				switch (fd.getType()) {
				case ENUM:
					res.add(((EnumValueDescriptor) field).getName());
					break;
				default:
					if (field instanceof String) {
						res.add(StringUtils.substring((String) field, 0, fieldDef.getLength()));
					} else {
						res.add(field);
					}
					break;
				}
				return res;
			}
			else throw new RuntimeException("Found a primitive type before the path end");
		}
	}

	private String getCondition(final String fieldPathCond) {
		return fieldPathCond.contains("[") ? StringUtils.substringAfter(fieldPathCond, "[").replace("]", "") : "";
	}
}
