package eu.dnetlib.data.bulktag;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Created by miriam on 02/08/2018.
 */
public class CommunityConfiguration {

    private static final Log log = LogFactory.getLog(CommunityConfiguration.class);

    enum MapModes{
        SUBJECT_MAP,
        DATASOURCE_MAP,
        ZENODO_COMMUNITY_MAP
    }

    private Map<String,Community> communities;

    //map subject -> communityid
    private transient Map<String,List<Pair<String,SelectionCriteria>>> subjectMap = new HashMap<>();
    //map datasourceid -> communityid
    private transient Map<String,List<Pair<String,SelectionCriteria>>> datasourceMap = new HashMap<>();
    //map zenodocommunityid -> communityid
    private transient Map<String,List<Pair<String,SelectionCriteria>>> zenodocommunityMap = new HashMap<>();

    public CommunityConfiguration(final Map<String, Community> communities) {
        this.communities = communities;
        init();
    }

    public void init() {

        if (subjectMap == null) {
            subjectMap = Maps.newHashMap();
        }
        if (datasourceMap == null) {
            datasourceMap = Maps.newHashMap();
        }
        if (zenodocommunityMap == null) {
            zenodocommunityMap = Maps.newHashMap();
        }

        for(Community c : getCommunities().values()) {
            //get subjects
            final String id = c.getId();
            for(String sbj : c.getSubjects()){
                Pair<String,SelectionCriteria> p = new Pair<>(id,new SelectionCriteria(null));
                add(sbj.toLowerCase().trim() , p, subjectMap);
            }
            //get datasources
            for(Datasource d: c.getDatasources()){
                add(d.getOpenaireId(),new Pair<>(id,d.getSelCriteria()),datasourceMap);
            }
            //get zenodo communities
            for(ZenodoCommunity zc : c.getZenodoCommunities()){
                add(zc.getZenodoCommunityId(),new Pair<>(id,zc.getSelCriteria()),zenodocommunityMap);
            }

        }
    }

    private void add(String key,Pair<String,SelectionCriteria> value, Map<String,List<Pair<String,SelectionCriteria>>> map){
        List<Pair<String,SelectionCriteria>> values = map.get(key);

        if (values == null){
            values = new ArrayList<>();
            map.put(key,values);
        }
        values.add(value);
    }

    public List<Pair<String,SelectionCriteria>> getCommunityForSubject(String sbj){
        return subjectMap.get(sbj);
    }

    public List<Pair<String,SelectionCriteria>> getCommunityForDatasource(String dts){
        return datasourceMap.get(dts);
    }

    public List<Pair<String,SelectionCriteria>> getCommunityForZenodoCommunity(String zc){
        return zenodocommunityMap.get(zc);
    }

    public List<String> getCommunityForSubjectValue(String value) {

        return getContextIds(subjectMap.get(value));
    }

    public List<String> getCommunityForDatasourceValue(String value) {

        return getContextIds(datasourceMap.get(value.toLowerCase()));
    }

    public List<String> getCommunityForZenodoCommunityValue(String value){

        return getContextIds(zenodocommunityMap.get(value.toLowerCase()));
    }

    private List<String> getContextIds(List<Pair<String, SelectionCriteria>> list) {
        if (list != null) {
            return list.stream().map(p -> p.getFst()).collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    /*
    public SelectionCriteria getSelCriteria(String value, String community, MapModes map_mode){

        Map<String,List<Pair<String,SelectionCriteria>>> map = null;
        if(map_mode == MapModes.DATASOURCE_MAP)
            map = datasourceMap;
        else
        if(map_mode == MapModes.ZENODO_COMMUNITY_MAP)
            map = zenodocommunityMap;
        else
            new Throwable("Impossible to have Selection Criteria over subjects");

        List<Pair<String, SelectionCriteria>> lst = map.get(value);
        List<SelectionCriteria> selectionList = lst.stream().map(p -> {
            if (p.fst == community)
                return p.snd;
            return null;
        }).collect(Collectors.toList());//for each community there will be only one Selection Criteria per datasource or zenodo community
        if(selectionList != null)
            if (selectionList.size()>0)
                return selectionList.get(0);
        return null;
    }
    */

    public Map<String, Community> getCommunities() {
        return communities;
    }

    public void setCommunities(Map<String, Community> communities) {
        this.communities = communities;
    }

    public String toJson() {
        final Gson g = new Gson();
        return g.toJson(this);
    }

    public int size() {
        return communities.keySet().size();
    }

    public Community getCommunityById(String id){
        return communities.get(id);
    }

    public List<Community> getCommunityList() {
        return Lists.newLinkedList(communities.values());
    }
}
