package eu.dnetlib.data.transform.xml;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.Author;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Metadata.Builder;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class OdfToHbaseXsltFunctions extends CommonDNetXsltFunctions {

	private static Map<String, String> mappingAccess = Maps.newHashMap();

	static {

		mappingAccess.put("info:eu-repo/semantics/openAccess", "OPEN");
		mappingAccess.put("info:eu-repo/semantics/closedAccess", "CLOSED");
		mappingAccess.put("info:eu-repo/semantics/restrictedAccess", "RESTRICTED");
		mappingAccess.put("info:eu-repo/semantics/embargoedAccess", "EMBARGO");

		// Transformator now maps the access rights into proper values, not sure if it does for all datasets.
		mappingAccess.put("OPEN", "OPEN");
		mappingAccess.put("CLOSED", "CLOSED");
		mappingAccess.put("RESTRICTED", "RESTRICTED");
		mappingAccess.put("EMBARGO", "EMBARGO");
		mappingAccess.put("OPEN SOURCE", "OPEN SOURCE");

	}

	public static String odfResult(
			final String resultId,
			final boolean invisible,
			final NodeList about,
			final NodeList metadata,
			final NodeList titles,
			final NodeList creators,
			final NodeList subjects,
			final NodeList publisher,
			final NodeList descriptions,
			final NodeList dates,
			final NodeList dateaccepted,
			final NodeList resourceTypes,
			final NodeList formats,
			final NodeList sizes,
			final NodeList languages,
			final NodeList cobjcategory,
			final NodeList contributors,
			final NodeList rights,
			final NodeList license,
			final NodeList version,
			final NodeList pidList,
			final String provenance,
			final String trust,
			final NodeList hostedby,
			final NodeList collectedfrom,
			final NodeList originalIds,
			final String instanceUri,
			final String landingPage,
			final NodeList distributionlocation,
			final NodeList documentationUrl,
			final String dateOfCollection,
			final String dateOfTransformation) {

		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();

			final Result.Builder result = Result.newBuilder();
			Result.Metadata.Builder metadataProto = Result.Metadata.newBuilder();

			// subject
			for (int i = 0; i < subjects.getLength(); i++) {
				Node currentNode = subjects.item(i);
				NodeList childNodes = currentNode.getChildNodes();
				if (childNodes.getLength() > 0) {
					String subjectValue = childNodes.item(0).getNodeValue();
					String schemeName = "keyword";
					String schemeURI ="keyword";
					if (currentNode.hasAttributes()) {
						NamedNodeMap attributes = currentNode.getAttributes();
						Node schemeNameNode = attributes.getNamedItem("subjectScheme");
						Node schemeURINode = attributes.getNamedItem("schemeURI");
						if(schemeNameNode != null) schemeName = schemeNameNode.getTextContent();
						if(schemeURINode != null) schemeURI = schemeURINode.getTextContent();
						if(schemeNameNode != null && schemeURINode == null) schemeURI = schemeName;
						if(schemeURINode != null && schemeNameNode == null) schemeName = schemeURI;
					}
					addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("subject"),
							getStructuredProperty(subjectValue, schemeURI, schemeName, "dnet:subject_classification_typologies", "dnet:subject_classification_typologies"));
				}
			}

			// title
			for (int i = 0; i < titles.getLength(); i++) {
				Node currentNode = titles.item(i);
				NodeList childNodes = currentNode.getChildNodes();
				if (childNodes.getLength() > 0) {
					String titleValue = childNodes.item(0).getNodeValue();
					String classname = "main title";
					String classid = "main title";
					if (currentNode.hasAttributes()) {
						NamedNodeMap attributes = currentNode.getAttributes();
						Node titleType = attributes.getNamedItem("titleType");

						if (titleType != null && titleType.getNodeValue().equals("AlternativeTitle")) {
							classname = "alternative title";
							classid = "alternative title";
						}
						if (titleType != null && titleType.getNodeValue().equals("Subtitle")) {
							classname = "subtitle";
							classid = "subtitle";
						}
						if (titleType != null && titleType.getNodeValue().equals("TranslatedTitle")) {
							classname = "translated title";
							classid = "translated title";
						}
					}
					addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("title"),
							getStructuredProperty(titleValue, classname, classid, "dnet:dataCite_title", "dnet:dataCite_title"));
				}
			}

			// creators
			for (int i = 0; i < creators.getLength(); i++) {
				final Element creator = (Element) creators.item(i);
				if (creator != null && creator.hasChildNodes()) {

					final NodeList creatorNames = creator.getElementsByTagName("creatorName");
					if (creatorNames.getLength() > 0) {
						createAuthor(metadataProto, i, creator, creatorNames);
					} else{
						//handle authors with namespaceprefix
						final NodeList creatorNamesNs = creator.getElementsByTagNameNS("http://datacite.org/schema/kernel-4", "creatorName");
						if (creatorNamesNs.getLength() > 0) {
							createAuthor(metadataProto, i, creator, creatorNamesNs);
						}

					}
				}
			}

			// description
			for (int i = 0; i < descriptions.getLength(); i++) {
				Element currentNode = (Element) descriptions.item(i);
				if (currentNode != null && currentNode.hasChildNodes()) {
					String descriptionValue = currentNode.getChildNodes().item(0).getNodeValue();

					final String descriptionType = currentNode.getAttribute("descriptionType");
					if (StringUtils.isNotBlank(descriptionType)) {
						switch (descriptionType) {
						case "TechnicalInfo":
							addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("tool"), descriptionValue);
							break;
						case "Abstract":
							addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("description"), descriptionValue);
							break;
						case "DistributionForm":
							addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("format"), descriptionValue);
							break;
						}
					} else {
						addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("description"), descriptionValue);
					}
				}
			}

			// contributors
			for (int i = 0; i < contributors.getLength(); i++) {
				final Element contributor = (Element) contributors.item(i);
				if (contributor != null && contributor.hasChildNodes()) {

					NodeList contributorNames = contributor.getElementsByTagName("contributorName");
					if (contributorNames != null) {
						Element contributorName = (Element) contributorNames.item(0);
						if (contributorName != null) {
							final String contributorValue = contributorName.getTextContent();
							final String contributorType = contributor.getAttribute("contributorType");

							if (StringUtils.isNotBlank(contributorType)) {
								switch (contributorType) {
								case "ContactPerson":
									addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("contactperson"), contributorValue);
									break;
								case "ContactGroup":
									addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("contactgroup"), contributorValue);
									break;
								}
							} else {
								addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("contributor"), contributorValue);
							}
						}
					}
				}
			}

			// publisher
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("publisher"), getFirstItem(publisher));



			// dates
			for (int i = 0; i < dates.getLength(); i++) {
				Node currentNode = dates.item(i);
				if (currentNode != null && currentNode.hasAttributes() && currentNode.hasChildNodes()) {
					String dateAttribute = currentNode.getAttributes().getNamedItem("dateType").getNodeValue();
					String dateValue = currentNode.getChildNodes().item(0).getNodeValue();
					String protoAttribute = "relevantdate";
					if ("Accepted".equals(dateAttribute)) {
						protoAttribute = "dateofacceptance";
					} else if ("Issued".equals(dateAttribute)) {
						protoAttribute = "storagedate";
					} else if ("Updated".equals(dateAttribute)) {
						protoAttribute = "lastmetadataupdate";
					} else if ("Available".equals(dateAttribute)) {
						protoAttribute = "embargoenddate";
					}
					if (protoAttribute.equals("relevantdate") == false) {
						addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName(protoAttribute), dateValue);
					} else {
						addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName(protoAttribute),
								getStructuredProperty(dateValue, "UNKNOWN", "UNKNOWN", "dnet:dataCite_date", "dnet:dataCite_date"));
					}
				}
			}

			//license
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("license"), getFirstItem(license));
			// dateofacceptance
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("dateofacceptance"), getFirstItem(dateaccepted));

			// size
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("size"), getFirstItem(sizes));

			// version
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("version"), getFirstItem(version));

			// language
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("language"),
					setQualifier(getDefaultQualifier("dnet:languages"), Lists.newArrayList(getFirstItem(languages))));

			// resource type
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("resourcetype"),
					setQualifier(getDefaultQualifier("dnet:dataCite_resource"), Lists.newArrayList(getFirstItem(resourceTypes))));

			// resultType
			final String cobjcategoryCode = getFirstItem(cobjcategory);
			final String resulttype = getResultType(cobjcategory);
			addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("resulttype"), getSimpleQualifier(resulttype, "dnet:result_typologies"));

			switch (resulttype) {
			case "software" :
				// format
				addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("programmingLanguage"),
						getSimpleQualifier(getFirstItem(formats), "dnet:programming_languages"));
				break;
			case "dataset":
				for (int i = 0; i < formats.getLength(); i++) {
					Node currentNode = formats.item(i);
					NodeList childNodes = currentNode.getChildNodes();
					if (childNodes.getLength() > 0) {
						String formatValue = childNodes.item(0).getNodeValue();
						addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("format"), formatValue);
					}
				}
				break;
			case "other":

				break;
			}

			// documentationUrl
			for (int i = 0; i < documentationUrl.getLength(); i++) {
				final Element docUrl = (Element) documentationUrl.item(i);
				if (docUrl != null && docUrl.hasChildNodes()) {
					final String value = docUrl.getTextContent();
					if (StringUtils.isNotBlank(value)) {
						addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("documentationUrl"), value);
					}
				}
			}

			ValueMap values = ValueMap.parseNodeList(metadata);
			// contexts
			if (values.get("concept") != null) {
				for (final eu.dnetlib.data.transform.xml.Element e : values.get("concept")) {
					final String id = e.getAttributes().get("id");
					if (StringUtils.isBlank(id)) throw new IllegalArgumentException("Context id cannot be blank");
					metadataProto.addContext(Context.newBuilder().setId(id));
				}
			}

			//journal
			if (values.containsKey("journal")) {
				for (final eu.dnetlib.data.transform.xml.Element journal : values.get("journal")) {
					addJournal(metadataProto, journal);

				}
			}

			final List<KeyValue> hostedBys = getKeyValues(ValueMap.parseNodeList(hostedby), "hostedby", Type.datasource);
			final List<KeyValue> collectedFroms = getKeyValues(ValueMap.parseNodeList(collectedfrom), "collectedfrom", Type.datasource);

			final Instance.Builder instance = Instance.newBuilder();

			String tmpRigths = "UNKNOWN";
			final String firstRight = getFirstItem(rights);
			if (mappingAccess.containsKey(firstRight)) {
				tmpRigths = mappingAccess.get(firstRight);
			}

			addField(instance, Instance.getDescriptor().findFieldByName("license"), getFirstItem(license));
			addField(instance, Instance.getDescriptor().findFieldByName("hostedby"), hostedBys);

			addField(instance, Instance.getDescriptor().findFieldByName("accessright"),
					setQualifier(getDefaultQualifier("dnet:access_modes"), Lists.newArrayList(tmpRigths)));

			addField(instance, Instance.getDescriptor().findFieldByName("instancetype"),
					setQualifier(getDefaultQualifier("dnet:dataCite_resource"), Lists.newArrayList(cobjcategoryCode)));

			if (StringUtils.isNotBlank(landingPage)) {
				addField(instance, Instance.getDescriptor().findFieldByName("url"), landingPage);
			}
			//sometimes the instanceUri is blank...
			if (StringUtils.isNotBlank(instanceUri)) {
				addField(instance, Instance.getDescriptor().findFieldByName("url"), instanceUri);
			}

			addField(instance, Instance.getDescriptor().findFieldByName("distributionlocation"), getFirstItem(distributionlocation));

			addField(instance, Instance.getDescriptor().findFieldByName("collectedfrom"), collectedFroms);
			addField(instance, Instance.getDescriptor().findFieldByName("dateofacceptance"), getFirstItem(dateaccepted));

			result.addInstance(instance);

			List<StructuredProperty> pids = parsePids(pidList);

			// original ids
			final Set<String> originalIdList = Sets.newHashSet();
			for (int i = 0; i < originalIds.getLength(); i++) {
				Node currentNode = originalIds.item(i);
				if (currentNode != null && currentNode.hasChildNodes()) {
					originalIdList.add(currentNode.getChildNodes().item(0).getNodeValue());
				}
			}

			OafEntity.Builder entity =
					getEntity(Type.result, entityId, collectedFroms, originalIdList, dateOfCollection, dateOfTransformation, pids).setResult(
							result.setMetadata(metadataProto));

			entity.setOaiprovenance(getOAIProvenance(about));

			Oaf oaf = getOaf(entity, getDataInfo(invisible, about, provenance, trust, false, false));
			return base64(oaf.toByteArray());
		} catch (Exception e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}

	}

	private static void createAuthor(final Builder metadataProto, final int i, final Element creator, final NodeList creatorNames) {
		final Element creatorName = (Element) creatorNames.item(0);

		final Author.Builder author = Author.newBuilder();
		author.setRank(i+1);
		final String fullname = StringUtils.trim(creatorName.getTextContent());

		author.setFullname(fullname);

		final eu.dnetlib.pace.model.Person p = new eu.dnetlib.pace.model.Person(fullname, false);
		if (p.isAccurate()) {
			author.setName(p.getNormalisedFirstName());
			author.setSurname(p.getNormalisedSurname());
		}
		final NodeList nameIdentifiers = creator.getElementsByTagName("nameIdentifier");
		if (nameIdentifiers.getLength() > 0) {
			final Element nameIdentifier = (Element) nameIdentifiers.item(0);
			final String nameIdentifierScheme = nameIdentifier.getAttribute("nameIdentifierScheme");
			final String id = StringUtils.trim(nameIdentifier.getTextContent());
			if (StringUtils.isNotBlank(id) && StringUtils.isNotBlank(nameIdentifierScheme)) {
				author.addPid(getKV(nameIdentifierScheme, id));
			}
		}

		addField(metadataProto, Result.Metadata.getDescriptor().findFieldByName("author"), author);
	}

	private static String getResultType(final NodeList cobjcategoryNode) {

		final ValueMap values = ValueMap.parseNodeList(cobjcategoryNode);

		final eu.dnetlib.data.transform.xml.Element cobjcategory = values.get("cobjcategory").stream()
				.map(e -> StringUtils.isNotBlank(e.getText()) ? e : new eu.dnetlib.data.transform.xml.Element("0000", e.getAttributes()))
				.findFirst()
				.orElse(new eu.dnetlib.data.transform.xml.Element("0000", new HashMap<>()));

		final String resulttype = cobjcategory.getAttributeValue("type");
		if (StringUtils.isNotBlank(resulttype)) {
			return resulttype;
		}

		return getDefaultResulttype(cobjcategory);
	}

	public static String getFirstItem(final NodeList list) {
		String out = "";
		if (list != null) {

			if (list.getLength() > 0 && list.item(0).getChildNodes() != null && list.item(0).getChildNodes().getLength() > 0) {
				out = list.item(0).getChildNodes().item(0).getNodeValue();
			}
		}
		return out;
	}

}
