package eu.dnetlib.msro.openaireplus.workflows.nodes;

import java.util.Iterator;
import java.util.Map;

import com.googlecode.sarasvati.Arc;
import com.googlecode.sarasvati.NodeToken;
import eu.dnetlib.common.logging.DnetLogger;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class IncrementalOperationJobNode extends SimpleJobNode {

    private static final Log log = LogFactory.getLog(IncrementalOperationJobNode.class);

    //incremental or refresh
    private String operationType;

    @Autowired
    private DnetLogger dnetLogger;

    @Override
    protected String execute(NodeToken nodeToken) throws Exception {

        if ("incremental".equalsIgnoreCase(operationType)) {
            final String currentWfProfileId = findCurrentWfProfileId(nodeToken);
            final Long lastSuccessStartDate = findLastSuccessStartDate(currentWfProfileId);
            if(forceRefresh(nodeToken, lastSuccessStartDate, currentWfProfileId)) return Arc.DEFAULT_ARC;
            log.info("Last success start date "+ lastSuccessStartDate);

            nodeToken.getFullEnv().setAttribute("OperationTypeInfo", "Operation type set to INCREMENTAL with date "+DateUtils.calculate_ISO8601(lastSuccessStartDate));
            nodeToken.getFullEnv().setAttribute("operationType", "INCREMENTAL");
            nodeToken.getFullEnv().setAttribute("DateFromFilter", lastSuccessStartDate);
            return Arc.DEFAULT_ARC;
        }
        nodeToken.getFullEnv().setAttribute("operationType", "REFRESH");
        nodeToken.getFullEnv().setAttribute("OperationTypeInfo", "Operation type manually set to REFRESH");
        return Arc.DEFAULT_ARC;
    }

    protected boolean forceRefresh(final NodeToken nodeToken, final Long lastSuccessStartDate, final String currentWfProfileId) throws Exception {
        if (lastSuccessStartDate < 0) {
            nodeToken.getFullEnv().setAttribute("OperationTypeInfo", "Last success start date < 0, operation forced to REFRESH");
            nodeToken.getFullEnv().setAttribute("operationType", "REFRESH");
            return true;
        }
        return false;
    }

    private Long findLastSuccessStartDate(String profId) {
        long res = -1;

        final Iterator<Map<String, String>> iter = dnetLogger.find(WorkflowsConstants.SYSTEM_WF_PROFILE_ID, profId);
        while (iter.hasNext()) {
            final Map<String, String> map = iter.next();
            if ("true".equalsIgnoreCase(map.get(WorkflowsConstants.SYSTEM_COMPLETED_SUCCESSFULLY))) {
                final long curr = NumberUtils.toLong(map.get(WorkflowsConstants.SYSTEM_START_DATE), -1);
                if (curr > res) {
                    res = curr;
                }
            }
        }
        log.debug("D-Net logger says the last success end date was on "+res);
        return res;
    }

    private String findCurrentWfProfileId(NodeToken token) throws MSROException {
        log.debug("Start to find the current profile Id");
        final String p1 = token.getEnv().getAttribute(WorkflowsConstants.SYSTEM_WF_PROFILE_ID);
        if (p1 != null && !p1.isEmpty()) {
            log.debug("The profile Id found is "+p1);
            return p1;
        }
        final String p2 = token.getFullEnv().getAttribute(WorkflowsConstants.SYSTEM_WF_PROFILE_ID);
        if (p2 != null && !p2.isEmpty()) {
            log.debug("The profile Id found is "+p2);
            return p2;
        }
        final String p3 = token.getProcess().getEnv().getAttribute(WorkflowsConstants.SYSTEM_WF_PROFILE_ID);
        if (p3 != null && !p3.isEmpty()) {
            log.debug("The profile Id found is "+p3);
            return p3;
        }
        throw new MSROException("Missing property in env: " + WorkflowsConstants.SYSTEM_WF_PROFILE_ID);
    }

    public String getOperationType() {
        return operationType;
    }

    public void setOperationType(final String operationType) {
        this.operationType = operationType;
    }
}
