package eu.dnetlib.domain.functionality;

import java.util.Date;

/**
 * This class represents an event generated by a notification query.
 * @author thanos@di.uoa.gr
 * @see eu.dnetlib.api.functionality.NotificationService
 * @see eu.dnetlib.functionality.notification.executor.NotificationQueryExecutor
 * @see NotificationQuery
 * @see NotificationSchedule
 * @see NotificationResult
 * @see NotificationSubscription
 * 
 */
public class NotificationEvent implements Comparable<NotificationEvent> {
	private String queryId;
	private Date date;

	/**
	 * Construct a new notification event.
	 * @param queryId the unique identifier of the query that generated this event
	 * @param date the date the event was generated
	 */
	public NotificationEvent(final String queryId, final Date date) {
		this.queryId = queryId;
		this.date = date;
	}
	
	/**
	 * Construct a new notification event with query identifier and date set to null.
	 */
	public NotificationEvent() {
		this(null, null);
	}
	
	/**
	 * Get the query of this event.
	 * @return the unique identifier of the query of this event
	 */
	public String getQueryId() {
		return queryId;
	}

	/**
	 * Set the query of this event. 
	 * @param queryId the unique identifier of the query of this event
	 */
	public void setQueryId(final String queryId) {
		this.queryId = queryId;
	}
	
	/**
	 * Get the date of this event.
	 * @return the date of this event 
	 */
	public Date getDate() {
		return date;
	}

	/**
	 * Set the date of this event.
	 * @param date the date of this event
	 */
	public void setDate(final Date date) {
		this.date = date;
	}
		
	@Override
	public int compareTo(final NotificationEvent event) {
		final int queryIdComparison = queryId.compareTo(event.queryId);
		final int dateComparison = date.compareTo(event.date);
		return (queryIdComparison == 0) ? dateComparison : queryIdComparison;
	}
	
	@Override
	public boolean equals(final Object object) {
		if (!(object instanceof NotificationEvent))
			return false;
		final NotificationEvent event = (NotificationEvent) object;
		return ((queryId == null) ? (event.queryId == null) : queryId.equals(event.queryId)) && ((date == null) ? (event.date == null) : date.equals(event.date));
	}
	
	@Override
	public int hashCode() {
		return ((queryId == null) ? 0 : queryId.hashCode()) + ((date == null) ? 0 : (date.hashCode()));
	}
	
	@Override
	public String toString() {
		return "(query: " + queryId + ", date: " + date + ")";
	}
}
