package eu.dnetlib.data.search.helpers.csv;

import eu.dnetlib.data.search.mappers.csv.ResearchProductCsvMapper;
import eu.dnetlib.dhp.schema.solr.RelatedRecord;
import eu.dnetlib.dhp.schema.solr.SolrRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Component
public class ResearchProductCsvConverter {

    @Autowired
    ResearchProductCsvMapper researchProductCsvMapper;

    public List<ResearchProductCsv> flattenProjects(SolrRecord solrRecord) {
        List<ResearchProductCsv> flattenedList = new ArrayList<>();

        if (solrRecord.getLinks() != null) {
            boolean hasProducedRecord = false;

            for (RelatedRecord relatedRecord : solrRecord.getLinks()) {
                if (isProducedByRelation(relatedRecord)) {
                    ResearchProductCsv newProduct = mapToResearchProductWithFunder(solrRecord, relatedRecord);
                    flattenedList.add(newProduct);
                    hasProducedRecord = true;
                }
            }

            // If no "isProducedBy" records found, map the entire solrRecord
            if (!hasProducedRecord) {
                ResearchProductCsv defaultProduct = researchProductCsvMapper.toResearchProductCsv(solrRecord);
                flattenedList.add(defaultProduct);
            }
        }

        return flattenedList;
    }

    private boolean isProducedByRelation(RelatedRecord relatedRecord) {
        return relatedRecord.getHeader() != null
                && "isProducedBy".equalsIgnoreCase(relatedRecord.getHeader().getRelationClass());
    }

    private ResearchProductCsv mapToResearchProductWithFunder(SolrRecord solrRecord, RelatedRecord relatedRecord) {
        ResearchProductCsv newProduct = researchProductCsvMapper.toSimpleResearchProductCsv(solrRecord);

        String funderString = Optional.ofNullable(relatedRecord.getFunding())
                .map(funding -> Optional.ofNullable(funding.getFunder())
                        .map(funder -> Optional.ofNullable(funder.getShortname())
                                .orElse(funder.getName())
                        ).orElse("")
                ).orElse("");

        String projectString = Optional.ofNullable(relatedRecord.getAcronym())
                .orElseGet(() -> Optional.ofNullable(relatedRecord.getProjectTitle())
                        .orElse("Unknown"));

        String funderProject = funderString + "|" + projectString;

        String code = Optional.ofNullable(relatedRecord.getCode())
                .filter(c -> !"unidentified".equalsIgnoreCase(c))
                .map(c -> "(" + c + ")")
                .orElse("");

        newProduct.setFunder_Project(funderProject + code);

        return newProduct;
    }
}
