package eu.dnetlib.data.search.mappers.csv;

import eu.dnetlib.data.search.helpers.csv.ProjectCsv;
import eu.dnetlib.dhp.schema.solr.Funding;
import eu.dnetlib.dhp.schema.solr.Project;
import eu.dnetlib.dhp.schema.solr.RelatedRecord;
import eu.dnetlib.dhp.schema.solr.SolrRecord;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Mapper(componentModel = "spring")
public interface ProjectCsvMapper {

    @Mapping(target="title", source="project.title")
    @Mapping(target="funder", expression="java(extractFunder(solrRecord))")
    @Mapping(target="startDate", source="project.startdate")
    @Mapping(target="endDate", source="project.enddate")
    @Mapping(target="participants", expression="java(extractParticipants(solrRecord))")
    ProjectCsv toProjectCsv(SolrRecord solrRecord);

    default String extractFunder(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord)
                .map(SolrRecord::getProject)
                .map(Project::getFunding)
                .map(Funding::getFunder)
                .map(funder ->{
                    String funderName = Optional.ofNullable(funder.getName()).orElse("");
                    String funderShortName = Optional.ofNullable(funder.getShortname()).orElse("");

                    return funderShortName.isEmpty()
                            ? funderName
                            : funderName + " (" + funderShortName + ")";
                }).orElse("");
    }

    default List<String> extractParticipants(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getLinks())
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(relatedRecord -> "hasParticipant".equals(relatedRecord.getHeader().getRelationClass()))
                .map(RelatedRecord::getLegalname)
                .filter(legalName -> legalName != null)
                .collect(Collectors.toList());
    }
}
