package eu.dnetlib.data.search.mappers.csv;

import eu.dnetlib.data.search.helpers.csv.ResearchProductCsv;
import eu.dnetlib.dhp.schema.solr.Author;
import eu.dnetlib.dhp.schema.solr.Pid;
import eu.dnetlib.dhp.schema.solr.SolrRecord;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Mapper(componentModel = "spring")
public interface ResearchProductCsvMapper {

    @Mapping(target="researchProduct", source = "result.resulttype")
    @Mapping(target="title", source="result.maintitle")
    @Mapping(target="authors", expression = "java(extractAuthor(solrRecord))")
    @Mapping(target = "publicationYear", source = "result.publicationdate", defaultValue = "")
    @Mapping(target = "doi", expression = "java(extractDoi(solrRecord))")
    @Mapping(target = "type", source = "result.resulttype")
    @Mapping(target = "peerReviewed", expression = "java(extractPeerReviewed(solrRecord))")
    @Mapping(target = "diamondJournal", source = "result.isInDiamondJournal")
    @Mapping(target = "journal", source = "result.journal.name", defaultValue = "")
    @Mapping(target = "downloadFrom", expression = "java(extractDownloadFrom(solrRecord))")
    @Mapping(target = "publiclyFunded", source = "result.publiclyFunded")
    @Mapping(target = "funder_Project", expression = "java(extractFunder(solrRecord))")
    @Mapping(target = "access", source = "result.bestaccessright.label")
    @Mapping(target = "green", source = "result.isGreen")
    @Mapping(target = "oaColour", source = "result.openAccessColor")
    ResearchProductCsv toResearchProductCsv(SolrRecord solrRecord);

    @Mapping(target="researchProduct", source = "result.resulttype")
    @Mapping(target="title", source="result.maintitle")
    @Mapping(target="authors", expression = "java(extractAuthor(solrRecord))")
    @Mapping(target = "publicationYear", source = "result.publicationdate", defaultValue = "")
    @Mapping(target = "doi", expression = "java(extractDoi(solrRecord))")
    @Mapping(target = "type", source = "result.resulttype")
    @Mapping(target = "peerReviewed", expression = "java(extractPeerReviewed(solrRecord))")
    @Mapping(target = "diamondJournal", source = "result.isInDiamondJournal")
    @Mapping(target = "journal", source = "result.journal.name", defaultValue = "")
    @Mapping(target = "downloadFrom", expression = "java(extractDownloadFrom(solrRecord))")
    @Mapping(target = "publiclyFunded", source = "result.publiclyFunded")
    @Mapping(target = "access", source = "result.bestaccessright.label")
    @Mapping(target = "green", source = "result.isGreen")
    @Mapping(target = "oaColour", source = "result.openAccessColor")
    ResearchProductCsv toSimpleResearchProductCsv(SolrRecord solrRecord);

    default List<String> extractAuthor(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getResult().getAuthor())
                .map(Collection::stream)
                .orElseGet(Stream::empty)
                .map(Author::getFullname)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    default String extractFunder(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getLinks())
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(relatedRecord -> "isProducedBy".equalsIgnoreCase(relatedRecord.getHeader().getRelationClass()))
                .map(relatedRecord -> {
                    //System.out.println("Funded!!!!!!!");
                    String funderString = Optional.ofNullable(relatedRecord.getFunding())
                            .map(funding -> Optional.ofNullable(funding.getFunder())
                                    .map(funder -> Optional.ofNullable(funder.getShortname())
                                            .orElse(funder.getName())
                                    ).orElse("")
                            ).orElse("");

                    String projectString = Optional.ofNullable(relatedRecord.getAcronym())
                            .orElseGet(() -> Optional.ofNullable(relatedRecord.getProjectTitle())
                                    .orElse("Unknown"));

                    funderString += "|" + projectString;

                    String code = Optional.ofNullable(relatedRecord.getCode())
                            .map(c -> "(" + c + ")")
                            .orElse("");

                    return funderString + code;
                })
                .collect(Collectors.joining(";"));
    }

    default List<String> extractDoi(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getPid())
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(pid -> "doi".equalsIgnoreCase(pid.getTypeCode()))
                .map(Pid::getValue)
                .collect(Collectors.toList());
    }

    default String extractPeerReviewed(SolrRecord solrRecord){
        return Optional.ofNullable(solrRecord.getResult().getInstance())
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(instance -> "peerReviewed".equalsIgnoreCase(instance.getRefereed()))
                .findFirst()
                .map(instance -> "yes")
                .orElse("no");
    }

    default String extractDownloadFrom(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getResult().getInstance())
                .map(List::stream) // Convert the instance list to a stream if it's not null
                .orElseGet(Stream::empty) // Provide an empty stream if the instance is null
                .filter(instance -> "OPEN".equalsIgnoreCase(instance.getAccessright().getCode())) // Filter for "OPEN"
                .findFirst() // Find the first matching instance
                .map(instance ->
                        Optional.ofNullable(instance.getUrl()) // Get the list of URLs
                                .filter(urls -> !urls.isEmpty()) // Ensure the list is not empty
                                .map(urls -> urls.get(0)) // Get the first URL from the list
                                .orElse("")) // Return empty string if the list is empty
                .orElseGet(() -> {
                    // Handle case if the first part didn't find a matching instance
                    return Optional.ofNullable(solrRecord.getResult().getInstance())
                            .map(List::stream) // Convert the instance list to a stream if it's not null
                            .orElseGet(Stream::empty) // Provide an empty stream if the instance is null
                            .findFirst() // Find the first instance
                            .map(instance ->
                                    Optional.ofNullable(instance.getUrl()) // Get the list of URLs
                                            .filter(urls -> !urls.isEmpty()) // Ensure the list is not empty
                                            .map(urls -> urls.get(0)) // Get the first URL from the list
                                            .orElse("")) // Return empty string if the list is empty
                            .orElse(""); // Return empty string if no instance is found
                });
    }
}
