package eu.dnetlib.data.search.mappers.html;

import eu.dnetlib.data.search.helpers.html.ResearchProductHtml;
import eu.dnetlib.dhp.schema.solr.Author;
import eu.dnetlib.dhp.schema.solr.Pid;
import eu.dnetlib.dhp.schema.solr.SolrRecord;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Mapper(componentModel = "spring")
public interface ResearchProductsHtmlMapper {
    @Mapping(target = "type", source = "result.resulttype")
    @Mapping(target = "title", source = "result.maintitle")
    @Mapping(target = "authors", expression = "java(extractAuthor(solrRecord))")
    @Mapping(target = "publicationYear", source = "result.publicationdate", defaultValue = "")
    @Mapping(target = "doi", expression = "java(extractDoi(solrRecord))")
    @Mapping(target = "permanentIdentifier", expression = "java(extractPermanentIdentifier(solrRecord))")
    @Mapping(target = "publicationType", expression = "java(exctractPublicationType(solrRecord))")
    @Mapping(target = "journal", source = "result.journal.name", defaultValue = "")
    @Mapping(target = "projectName_gaNumber", expression = "java(extractFunder(solrRecord))")
    @Mapping(target = "accessMode", source = "result.bestaccessright.label")
    ResearchProductHtml toResearchProductHtml(SolrRecord solrRecord);

    default String extractAuthor(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getResult().getAuthor())
                .map(authors -> authors.stream()
                        .map(Author::getFullname)
                        .filter(Objects::nonNull)
                        .collect(Collectors.joining(";")))
                .orElse("");
    }

    default String extractPermanentIdentifier(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getResult().getInstance())
                .map(List::stream) // Convert the instance list to a stream if it's not null
                .orElseGet(Stream::empty) // Provide an empty stream if the instance is null
                .findFirst() // Find the first instance
                .map(instance ->
                        Optional.ofNullable(instance.getUrl()) // Get the list of URLs
                                .filter(urls -> !urls.isEmpty()) // Ensure the list is not empty
                                .map(urls -> urls.get(0)) // Get the first URL from the list
                                .orElse("")) // Return empty string if the list is empty
                .orElse("");
    }

    default String exctractPublicationType(SolrRecord solrRecord){
        return Optional.ofNullable(solrRecord.getResult().getInstance())
                .map(instances -> instances.stream()
                        .findFirst()
                        .map(instance -> instance.getInstancetype()).orElse(""))
                .orElse("");
    }

    default String extractDoi(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getPid())
                .map(pids -> pids.stream()
                        .filter(pid -> "doi".equalsIgnoreCase(pid.getTypeCode()))
                        .map(Pid::getValue)
                        .collect(Collectors.joining(";")))
                .orElse("");
    }

    default String extractFunder(SolrRecord solrRecord) {
        return Optional.ofNullable(solrRecord.getLinks())
                .map(List::stream)
                .orElseGet(Stream::empty)
                .filter(relatedRecord -> "isProducedBy".equalsIgnoreCase(relatedRecord.getHeader().getRelationClass()))
                .map(relatedRecord -> {
                    //System.out.println("Funded!!!!!!!");
                    String funderString = Optional.ofNullable(relatedRecord.getFunding())
                            .map(funding -> Optional.ofNullable(funding.getFunder())
                                    .map(funder -> Optional.ofNullable(funder.getShortname())
                                            .orElse(funder.getName())
                                    ).orElse("")
                            ).orElse("");

                    String projectString = Optional.ofNullable(relatedRecord.getAcronym())
                            .orElseGet(() -> Optional.ofNullable(relatedRecord.getProjectTitle())
                                    .orElse("Unknown"));

                    funderString += "|" + projectString;

                    String code = Optional.ofNullable(relatedRecord.getCode())
                            .map(c -> "(" + c + ")")
                            .orElse("");

                    return funderString + code;
                })
                .collect(Collectors.joining(";"));
    }
}