package eu.dnetlib.data.search.solr;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.dataformat.csv.CsvSchema;
import com.google.gson.Gson;
import eu.dnetlib.api.data.SearchServiceException;
import eu.dnetlib.data.search.helpers.csv.*;
import eu.dnetlib.data.search.helpers.html.ResearchProductHtml;
import eu.dnetlib.data.search.mappers.csv.DatasourceCsvMapper;
import eu.dnetlib.data.search.mappers.csv.OrganizationCsvMapper;
import eu.dnetlib.data.search.mappers.csv.ProjectCsvMapper;
import eu.dnetlib.data.search.mappers.csv.ResearchProductCsvMapper;
import eu.dnetlib.data.search.mappers.html.ResearchProductsHtmlMapper;
import eu.dnetlib.data.search.transform.Transformer;
import eu.dnetlib.data.search.transform.TransformerException;
import eu.dnetlib.data.search.utils.solr.SolrResultSetOptionsUtil;
import eu.dnetlib.data.search.utils.solr.SolrResultsFormatter;
import eu.dnetlib.dhp.schema.solr.*;
import eu.dnetlib.domain.EPR;
import gr.uoa.di.driver.enabling.resultset.ResultSet;
import io.micrometer.core.instrument.Timer;
import io.micrometer.prometheus.PrometheusMeterRegistry;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.log4j.Logger;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.params.SolrParams;
import org.apache.solr.common.util.NamedList;
import org.z3950.zing.cql.CQLParseException;

import javax.ws.rs.core.MediaType;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;

/**
 * Created by antleb on 2/4/14.
 */

public class SolrResultSet implements ResultSet<String> {

    private Logger logger = Logger.getLogger(getClass());

    private EPR epr = null;
    public SolrClient solrClient = null;

    private NamedList<String> queryOpts = new NamedList<String>();
    long size = -1;

    final ObjectMapper objectMapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

    private PrometheusMeterRegistry registry;

    DatasourceCsvMapper datasourceCsvMapper;
    OrganizationCsvMapper organizationCsvMapper;
    ProjectCsvMapper projectCsvMapper;
    ResearchProductCsvMapper researchProductCsvMapper;
    ResearchProductCsvConverter researchProductCsvConverter;
    ResearchProductsHtmlMapper researchProductsHtmlMapper;

    CsvMapper datasourceMapper = new CsvMapper();
    CsvMapper organizationMapper = new CsvMapper();
    CsvMapper projectMapper = new CsvMapper();
    CsvMapper researchProductMapper = new CsvMapper();

    public SolrResultSet(EPR epr, CloudSolrClient cloudSolrClient, PrometheusMeterRegistry registry, DatasourceCsvMapper
                         datasourceCsvMapper, OrganizationCsvMapper organizationCsvMapper, ProjectCsvMapper projectCsvMapper,
                         ResearchProductCsvMapper researchProductCsvMapper, ResearchProductCsvConverter researchProductCsvConverter,
                         ResearchProductsHtmlMapper researchProductsHtmlMapper)
            throws IOException, CQLParseException, SearchServiceException {
        this.epr = epr;
        this.solrClient = cloudSolrClient;

        this.queryOpts = SolrResultSetOptionsUtil.extractQueryOptions(epr.getParameter("query"));

        String layout = epr.getParameter("layout");
        String mdformat = epr.getParameter("mdformat");
        String interpretation = epr.getParameter("interpretation");

        this.datasourceCsvMapper = datasourceCsvMapper;
        this.organizationCsvMapper = organizationCsvMapper;
        this.projectCsvMapper = projectCsvMapper;
        this.researchProductCsvMapper = researchProductCsvMapper;
        this.researchProductCsvConverter = researchProductCsvConverter;
        this.researchProductsHtmlMapper = researchProductsHtmlMapper;

        System.out.println("SOLR CLIENT " + solrClient);
        ((CloudSolrClient)solrClient).setDefaultCollection(mdformat + "-" + layout + "-" + interpretation);

        this.registry = registry;
    }

    @Override
    public boolean isOpen() {
        return true;
    }

    @Override
    public boolean isAlive() {
        return true;
    }

    @Override
    public void close() {

        /* Closes the whole pull. We do not want this.
        try {
            logger.info("!!!!!!!!! !!!!!!! CLOSING !!!!!!!!! !!!!!!!!!! ");
            solrClient.close();

        } catch (IOException e) {
            logger.error("Error closing result set.", e);
        }*/
    }

    @Override
    public int size() {
        return (int) size;
    }

    @Override
    @Deprecated
    public List<String> getElements(int from, int to) {
        return get(from, to);
    }

    List<FacetField> facetFields = null;

    @Override
    @Deprecated
    public List<String> get(int from, int to) {
        List<String> res = new ArrayList<String>();

        QueryResponse rsp = null;

        HashMap<String, List<String>> map = new HashMap<String, List<String>>();

        logger.debug("from: " + from);
        logger.debug("to: " + to);


        queryOpts.add("start", from+1 + "");
        queryOpts.add("rows", to + 1+"");

        try {

            rsp = solrClient.query(SolrParams.toSolrParams(queryOpts));

            facetFields = rsp.getFacetFields();
            SolrDocumentList docs = rsp.getResults();

            if (facetFields!=null && !facetFields.isEmpty()) {
                for (int i = from - 1; i < to; i++) {
                    for (FacetField field : facetFields) {
                        if (field.getValueCount() > i) {
                            BrowseField bf = new BrowseField();
                            bf.setId(field.getValues().get(i).getName());
                            bf.setName(field.getValues().get(i).getName());
                            bf.setCount(field.getValues().get(i).getCount() + "");
                            if (map.get(field.getName()) == null) {
                                map.put(field.getName(), new ArrayList<String>());
                            }

                            map.get(field.getName()).add(new Gson().toJson(bf));
                        }
                    }
                }

                for (Map.Entry<String, List<String>> facetEntry : map.entrySet()) {
                    StringBuilder builder = new StringBuilder();
                    builder.append("\"" + facetEntry.getKey() + "\"" + " : ");
                    builder.append(facetEntry.getValue());
                    res.add(builder.toString());
                }
            }

            logger.debug("time: " + rsp.getElapsedTime());
            logger.debug("found: " + docs.getNumFound());
            logger.debug("docs: " + docs.size());

            for (int i = 0; i < docs.size(); i++) {
                String result = (String) docs.get(i).get("__result");
                res.add(result);
            }

            return res;

        } catch (SolrServerException sse) {
            logger.error("Fail to get results from Solr. ", sse);

        } catch (IOException ioe) {
            logger.error("Fail to get results from Solr. ", ioe);
        }

        return null;
    }

    @Override
    public EPR getEpr() {
        return null;
    }

    public Map<String,List<String>> newGet(int from, int to, String format, Transformer transformer,
                                            Transformer oldRefineTransformer, boolean timeout, String timeoutValue) throws SearchServiceException {
        List<String> refineSolrResults = new ArrayList<String>();
        List<String> searchSolrResults = new ArrayList<String>();

        QueryResponse rsp = null;
        HashMap<String, List<String>> map = new HashMap<String, List<String>>();

        if (from < 0) from = 0;
        if (to < 0) to = 0;
        queryOpts.add("start", from*to + "");
        queryOpts.add("rows", to +"");

        try {
            if (timeout) {
                if (timeoutValue != null && Integer.valueOf(timeoutValue) == -1) {
                    logger.warn("Timeout value is -1. Timeout will not be set for query: " + queryOpts.get("q"));
                    timeout = false;
                } else if (Integer.valueOf(timeoutValue) < 500) {
                    logger.warn("Timeout is less than 500ms. We fall back to the default 1000ms for query:" + queryOpts.get("q"));
                    queryOpts.add("timeAllowed", "500");
                } else if (timeout) {
                    queryOpts.add("timeAllowed", timeoutValue);
                }
            }
        } catch (NumberFormatException nfe) {
            logger.warn("NumberFormatException occurred. Timeout not properly set. We fall back to the default 1000ms.");
            queryOpts.add("timeAllowed", "500");
        }

        logger.info("timeout: " + timeout + ", timeout value: " + queryOpts.get("timeAllowed"));

        //queryOpts.add("f.resulthostingdatasource.facet.limit", "2");

        long startTime = System.nanoTime();

        try {
            io.micrometer.core.instrument.Timer.Sample timer = Timer.start(registry);
            rsp = solrClient.query(SolrParams.toSolrParams(queryOpts));
            timer.stop(registry.timer("solr.server.response.duration"));

            long estimatedTime = System.nanoTime() - startTime;
            logger.info("Solrj time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") +
                    " and sorts " + queryOpts.getAll("sort") + " from: "
                    + from + " and size " + to + " with timeout: " + timeout);

            /*System.out.println("Solrj time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") +
                    " and sorts " + queryOpts.getAll("sort") + " from: "
                    + from + " and size " + to);
            */
            facetFields = rsp.getFacetFields();

            SolrDocumentList docs = rsp.getResults();

            this.size = docs.getNumFound();

            if (facetFields!=null && !facetFields.isEmpty()) {
                if (format != null && format.equals(MediaType.APPLICATION_JSON)) {
                    for (FacetField field : facetFields) {
                        map.put(field.getName(), new ArrayList<String>());
                        BrowseField bf = null;
                        for (int i = 0; i < field.getValueCount(); i++) {
                            bf = new BrowseField();
                            //bf.setId(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName()));
                            bf.setId(field.getValues().get(i).getName());
                            String[] facetedValues = field.getValues().get(i).getName().split("\\|\\|",2);


                            logger.debug("faceted values " + Arrays.toString(facetedValues));

                            if (facetedValues.length > 1) {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(facetedValues[1]));
                                bf.setName(facetedValues[1]);
                                logger.debug("faceted values [1] " + facetedValues[1]);

                            } else if (field.getValues().get(i).getName().split("_\\:\\:",2).length > 1) {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName().split("\\:\\:",2)[1]).replaceAll("\\:\\:", "\\|"));
                                bf.setName(field.getValues().get(i).getName().split("\\:\\:",2)[1].replaceAll("\\:\\:", "\\|"));

                            } else {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName()));
                                bf.setName(field.getValues().get(i).getName());
                            }

                            bf.setCount(field.getValues().get(i).getCount() + "");
                            map.get(field.getName()).add(new Gson().toJson(bf));
                        }

                    }

                    StringBuilder builder = null;

                    for (Map.Entry<String, List<String>> facetEntry : map.entrySet()) {
                        builder = new StringBuilder();
                        builder.append("\"" + facetEntry.getKey() + "\"" + " : ");
                        builder.append(facetEntry.getValue());
                        refineSolrResults.add(builder.toString());
                    }

                } else { //the old implementation & xml as default
                    logger.debug("Creating old browse results.");
                    createXmlRefineFields(refineSolrResults, oldRefineTransformer);
                }
            }

            startTime = System.nanoTime();

            for (int i = 0; i < docs.size(); i++) {
                String result = (String) docs.get(i).get("__result");
                //logger.debug("["+ i +"]: " + docs.get(i).get("__result"));

                try {
                    if (transformer != null) {
                        //logger.debug("1 >>>>>>" + result);
                        String xml = result.replaceAll("<em>","").replaceAll("</em>","");
                        result = transformer.transform(xml);
                        //logger.debug("2 >>>>>>" + result);
                    }

                } catch (TransformerException te) {
                    logger.warn("Error transforming " + result, te);
                    continue;
                }

                if (format != null && format.equals(MediaType.APPLICATION_JSON)) {
                    searchSolrResults.add(SolrResultsFormatter.xml2Json(result));
                } else { // default xml
                    searchSolrResults.add(result);
                }
            }

            estimatedTime = System.nanoTime() - startTime;
            logger.info("Internal transformation time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: "
                    + from + " and size " + to);

            Map<String,List<String>> response = new HashMap<String, List<String>>();

            //logger.debug("refine results " + refineSolrResults);
            //logger.info("search results SIZE " + searchSolrResults.size());
            //logger.info("search results " + searchSolrResults);


            response.put("refine",refineSolrResults);
            response.put("search", searchSolrResults);

            return response;

        } catch (SolrServerException sse) {
            logger.error("Error calling Solr for  query:" + queryOpts.get("q") + " and facets "
                    + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: " + from +
                    " and size " + to, sse);
            throw new SearchServiceException("Error contacting Solr.", sse);

        } catch (IOException ioe) {
            logger.error("Error calling Solr for  query:" + queryOpts.get("q") + " and facets "
                    + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: " + from +
                    " and size " + to, ioe);
            throw new SearchServiceException("Error reading results from Solr.", ioe);

        }
    }


    public Map<String,List<String>> newJsonGet(int from, int to, String format, Transformer transformer,
                                           Transformer oldRefineTransformer, boolean timeout, String timeoutValue) throws SearchServiceException {
        List<String> refineSolrResults = new ArrayList<String>();
        List<String> searchSolrResults = new ArrayList<String>();

        QueryResponse rsp = null;
        HashMap<String, List<String>> map = new HashMap<String, List<String>>();

        if (from < 0) from = 0;
        if (to < 0) to = 0;
        queryOpts.add("start", from*to + "");
        queryOpts.add("rows", to +"");

        try {
            if (timeout) {
                if (timeoutValue != null && Integer.valueOf(timeoutValue) == -1) {
                    logger.warn("Timeout value is -1. Timeout will not be set for query: " + queryOpts.get("q"));
                    timeout = false;
                } else if (Integer.valueOf(timeoutValue) < 500) {
                    logger.warn("Timeout is less than 500ms. We fall back to the default 1000ms for query:" + queryOpts.get("q"));
                    queryOpts.add("timeAllowed", "500");
                } else if (timeout) {
                    queryOpts.add("timeAllowed", timeoutValue);
                }
            }
        } catch (NumberFormatException nfe) {
            logger.warn("NumberFormatException occurred. Timeout not properly set. We fall back to the default 1000ms.");
            queryOpts.add("timeAllowed", "500");
        }

        logger.info("timeout: " + timeout + ", timeout value: " + queryOpts.get("timeAllowed"));

        //queryOpts.add("f.resulthostingdatasource.facet.limit", "2");

        long startTime = System.nanoTime();

        try {
            io.micrometer.core.instrument.Timer.Sample timer = Timer.start(registry);
            rsp = solrClient.query(SolrParams.toSolrParams(queryOpts));
            timer.stop(registry.timer("solr.server.response.duration"));

            long estimatedTime = System.nanoTime() - startTime;
            logger.info("Solrj time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") +
                    " and sorts " + queryOpts.getAll("sort") + " from: "
                    + from + " and size " + to + " with timeout: " + timeout);

            /*System.out.println("Solrj time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") +
                    " and sorts " + queryOpts.getAll("sort") + " from: "
                    + from + " and size " + to);
            */
            facetFields = rsp.getFacetFields();

            SolrDocumentList docs = rsp.getResults();

            this.size = docs.getNumFound();

            if (facetFields!=null && !facetFields.isEmpty()) {
                if (format != null && format.equals(MediaType.APPLICATION_JSON)) {
                    for (FacetField field : facetFields) {
                        map.put(field.getName(), new ArrayList<String>());
                        BrowseField bf = null;
                        for (int i = 0; i < field.getValueCount(); i++) {
                            bf = new BrowseField();
                            //bf.setId(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName()));
                            bf.setId(field.getValues().get(i).getName());
                            String[] facetedValues = field.getValues().get(i).getName().split("\\|\\|",2);
                            logger.debug("faceted values " + Arrays.toString(facetedValues));

                            if (facetedValues.length > 1) {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(facetedValues[1]));
                                bf.setName(facetedValues[1]);
                                logger.debug("faceted values [1] " + facetedValues[1]);

                            } else if (field.getValues().get(i).getName().split("_\\:\\:",2).length > 1) {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName().split("\\:\\:",2)[1]).replaceAll("\\:\\:", "\\|"));
                                bf.setName(field.getValues().get(i).getName().split("\\:\\:",2)[1].replaceAll("\\:\\:", "\\|"));

                            } else {
                                //bf.setName(org.apache.commons.lang3.StringEscapeUtils.escapeJson(field.getValues().get(i).getName()));
                                bf.setName(field.getValues().get(i).getName());
                            }

                            bf.setCount(field.getValues().get(i).getCount() + "");
                            map.get(field.getName()).add(new Gson().toJson(bf));
                        }

                    }

                    StringBuilder builder = null;

                    for (Map.Entry<String, List<String>> facetEntry : map.entrySet()) {
                        builder = new StringBuilder();
                        builder.append("\"" + facetEntry.getKey() + "\"" + " : ");
                        builder.append(facetEntry.getValue());
                        refineSolrResults.add(builder.toString());
                    }

                } else { //the old implementation & xml as default
                    logger.debug("Creating old browse results.");
                    createXmlRefineFields(refineSolrResults, oldRefineTransformer);
                }
            }

            startTime = System.nanoTime();

            for (int i = 0; i < docs.size(); i++) {
                String result = (String) docs.get(i).get("__json");

                //System.out.println("\n\n\n\n\n JSON ---------------\n"+ result);
                //logger.debug("["+ i +"]: " + docs.get(i).get("__result"));

                /*try {
                    if (transformer != null) {
                        //logger.debug("1 >>>>>>" + result);
                        String xml = result.replaceAll("<em>","").replaceAll("</em>","");
                        result = transformer.transform(xml);
                        //logger.debug("2 >>>>>>" + result);
                    }

                } catch (TransformerException te) {
                    logger.warn("Error transforming " + result, te);
                    continue;
                }

                if (format != null && format.equals(MediaType.APPLICATION_JSON)) {
                    searchSolrResults.add(SolrResultsFormatter.xml2Json(result));
                } else { // default xml*/
                //System.out.println("Adding result");
               // System.out.println("RESULT " + result);
                searchSolrResults.add(result);
                /*}*/
            }

            estimatedTime = System.nanoTime() - startTime;
            logger.info("Internal transformation time " + estimatedTime/1000000 +  " milliseconds for query:" + queryOpts.get("q") +
                    " and facets " + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: "
                    + from + " and size " + to);

            Map<String,List<String>> response = new HashMap<String, List<String>>();

            //logger.debug("refine results " + refineSolrResults);
            //logger.info("search results SIZE " + searchSolrResults.size());
            //logger.info("search results " + searchSolrResults);


            response.put("refine",refineSolrResults);
            response.put("search", searchSolrResults);

            return response;

        } catch (SolrServerException sse) {
            logger.error("Error calling Solr for  query:" + queryOpts.get("q") + " and facets "
                    + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: " + from +
                    " and size " + to, sse);
            throw new SearchServiceException("Error contacting Solr.", sse);

        } catch (IOException ioe) {
            logger.error("Error calling Solr for  query:" + queryOpts.get("q") + " and facets "
                    + queryOpts.getAll("facet.field") + " and fq " + queryOpts.getAll("fq") + " from: " + from +
                    " and size " + to, ioe);
            throw new SearchServiceException("Error reading results from Solr.", ioe);

        }
    }

    /**
     * limit is the maximum number of results the cursor get is allowed to fetch. If limit is set to -1 all
     * results are returned.
     */
    public void cursorGet(Transformer transformer, boolean special, boolean hasTitle, boolean isHtml,
                          int limit, OutputStream os) throws SolrServerException, SearchServiceException {

        System.out.println("Datasource Mapper " + datasourceCsvMapper);

        int rows = 500;
        int limitCounter = (limit > 0) ? limit / rows : -1;
        int headerCounter = (limit > 0) ? limit / rows : -1;

        setupQueryOptions(rows);

        String cursorMark = "*";
        String nextCursorMark = "";

        try {
            QueryResponse resp = null;
            if(isHtml) initializeHtmlTable(os);

            do {
                resp = solrClient.query(SolrParams.toSolrParams(queryOpts));
                nextCursorMark = resp.getNextCursorMark();

                for (int i = 0; i < resp.getResults().size(); i++) {
                    SolrRecord record = objectMapper.readValue((String) resp.getResults().get(i).get("__json"),
                            new TypeReference<SolrRecord>() {});

                    if(!isHtml) {
                        CsvEntity csvEntity = createCsvEntity(record);
                        CsvMapper csvMapper = selectCsvMapper(record);

                        if (hasTitle && i==0 && limitCounter == headerCounter) writeHeader(csvMapper, csvEntity, os);

                        if (special) {
                            List<ResearchProductCsv> flattenedProducts = researchProductCsvConverter.flattenProjects(record);
                            CsvSchema schema = csvMapper.schemaFor(csvEntity.getClass()).withoutHeader();
                            ObjectWriter csvWriterRow = researchProductMapper.writer(schema);
                            for (ResearchProductCsv internalCsvEntity : flattenedProducts) {

                                String csvRow = csvWriterRow.writeValueAsString(internalCsvEntity);
                                //System.out.println("CSV ROW special" + csvRow);
                                os.write(csvRow.getBytes());
                                os.flush();
                            }

                        } else {
                            if (csvEntity != null && csvMapper != null)
                                writeCsvEntity(csvMapper, csvEntity, special, os);
                        }

                    } else {
                        writeHtmlRow(os, record);
                    }
                }

                limitCounter--;
                updateCursorMark(nextCursorMark);

            } while (!cursorMark.equals(nextCursorMark) && limitCounter != 0);

        } catch (IOException ioe) {
            logger.error("Error executing solr query. ", ioe);
        }
    }

    private void writeHtmlRow(OutputStream os, SolrRecord record) throws IOException {
        StringBuilder htmlBuilder = new StringBuilder();
        ResearchProductHtml researchProductHtml = researchProductsHtmlMapper.toResearchProductHtml(record);
        htmlBuilder.append("<tr>")
                .append("<td>").append(researchProductHtml.getType()).append("</td>")
                .append("<td>").append(researchProductHtml.getTitle()).append("</td>")
                .append("<td>").append(researchProductHtml.getAuthors()).append("</td>")
                .append("<td>").append(researchProductHtml.getPublicationYear()).append("</td>")
                .append("<td>").append(researchProductHtml.getDoi()).append("</td>")
                .append("<td>").append(researchProductHtml.getPermanentIdentifier()).append("</td>")
                .append("<td>").append(researchProductHtml.getPublicationType()).append("</td>")
                .append("<td>").append(researchProductHtml.getJournal()).append("</td>")
                .append("<td>").append(researchProductHtml.getProjectName_gaNumber()).append("</td>")
                .append("<td>").append(researchProductHtml.getAccessMode()).append("</td>")
                .append("</tr>");
        os.write(htmlBuilder.toString().getBytes());
        os.flush();
    }

    private CsvEntity createCsvEntity(SolrRecord solrRecord) {
        Datasource datasource = solrRecord.getDatasource();
        Project project = solrRecord.getProject();
        Result result = solrRecord.getResult();
        Organization organization = solrRecord.getOrganization();

        if (datasource != null) return datasourceCsvMapper.toDatasourceCsv(solrRecord);
        if (project != null) return projectCsvMapper.toProjectCsv(solrRecord);
        if (result != null) return researchProductCsvMapper.toResearchProductCsv(solrRecord);
        if (organization != null) return organizationCsvMapper.toOrganizationCsv(solrRecord);

        return null;
    }


    private CsvMapper selectCsvMapper(SolrRecord solrRecord) {
        Datasource datasource = solrRecord.getDatasource();
        Project project = solrRecord.getProject();
        Result result = solrRecord.getResult();
        Organization organization = solrRecord.getOrganization();

        if (datasource != null) return datasourceMapper;
        if (project != null) return projectMapper;
        if (result != null) return researchProductMapper;
        if (organization != null) return organizationMapper;

        return null;
    }


    private static void initializeHtmlTable(OutputStream os) throws IOException {
        StringBuilder htmlBuilder = new StringBuilder();
        // Start the HTML table and headers
        htmlBuilder.append("<table>")
                .append("<thead><tr>")
                .append("<th>Type</th>" +
                        "<th>Title</th>" +
                        "<th>Authors</th>" +
                        "<th>Publication Year</th>" +
                        "<th>DOI</th>" +
                        "<th>Permanent Identifier</th>" +
                        "<th>Publication type</th>" +
                        "<th>Journal</th>" +
                        "<th>Project Name (GA Number)</th>" +
                        "<th>Access Mode</th>") // Add more headers as needed
                .append("</tr></thead><tbody>");
        os.write(htmlBuilder.toString().getBytes());
        os.flush();
    }

    public void writeHeader(CsvMapper csvMapper, CsvEntity csvEntity, OutputStream os) throws IOException {
        CsvSchema schema = csvMapper.schemaFor(csvEntity.getClass()).withHeader();
        ObjectWriter csvWriterHeader = csvMapper.writer(schema);
        String csvHeader = csvWriterHeader.writeValueAsString(csvEntity.getClass()).split(System.lineSeparator())[0];
        os.write(csvHeader.getBytes());
        os.write("\n".getBytes());
        os.flush();

    }

    //TODO add special
    public void writeCsvEntity(CsvMapper csvMapper, CsvEntity csvEntity, boolean special, OutputStream os) throws IOException {
        CsvSchema schema = csvMapper.schemaFor(csvEntity.getClass()).withoutHeader();
        ObjectWriter csvWriterRow = csvMapper.writer(schema.withoutHeader());
        String csvRow = csvWriterRow.writeValueAsString(csvEntity);

        os.write(csvRow.getBytes());
        os.flush();
    }

    public void updateCursorMark(String nextCursorMark) {
        queryOpts.remove("cursorMark");
        queryOpts.add("cursorMark", nextCursorMark);
    }

    private void setupQueryOptions(int rows) {
        queryOpts.add("start", "0");
        queryOpts.add("rows", "0");
        queryOpts.remove("rows");
        queryOpts.add("rows", rows +"");
        //queryOpts.add("fl", "__result");
        queryOpts.add("shards.tolerant","true");
        queryOpts.add("cursorMark", "*");
        queryOpts.add("sort", "__indexrecordidentifier asc");
    }

    //TODO get rid of this as soon as Joomla portal is out
    //Just copied and refactored the old one...
    @Deprecated
    private void createXmlRefineFields(List<String> res, Transformer oldRefineTransformer) {
        int max = -12;

        for (FacetField field:facetFields) {
            logger.debug("field " + field.getName() + " has count " + field.getValueCount());

            if (field.getValueCount() > max) {
                max = field.getValueCount();
            }
        }

        logger.debug("max " + max);

        for (int i = 0; i < max; i++) {
            StringBuilder sb = new StringBuilder();

            sb.append("<row>");
            for (FacetField field:facetFields) {
                if (field.getValueCount() > i) {
                    sb.append("<groupresult field=\"").append(field.getName()).append("\">");
                    sb.append("<count>").append(field.getValues().get(i).getCount()).append("</count>");
                    sb.append("<originalValue>").append(StringEscapeUtils.escapeXml(field.getValues().get(i).getName())).append("</originalValue>");

                    String[] facetValues = field.getValues().get(i).getName().split("\\|\\|");
                    if(facetValues.length > 1) {
                        sb.append("<value>").append(StringEscapeUtils.escapeXml(facetValues[1])).append("</value>");
                    } else {
                        sb.append("<value>").append(StringEscapeUtils.escapeXml(facetValues[0])).append("</value>");
                    }
                    sb.append("</groupresult>");
                }
            }
            sb.append("</row>");

            try {
                //logger.debug("row: " + sb.toString());
                //logger.debug("row2: " + oldRefineTransformer.transform(sb.toString()));

                //TODO remove
                res.add(oldRefineTransformer.transform(sb.toString()));

            } catch (TransformerException te) {
                logger.error("Cannot transform refine for: " + sb.toString(), te);
            }
        }
    }
    public static void main(String[] args) throws IOException {
        String json = "{\"header\":{\"id\":\"07b5c0ccd4fe::6cc02bea6ffd14eb21e6de7fa7e4271d\",\"originalId\":[\"07b5c0ccd4fe::2412-8368\"],\"recordType\":\"datasource\",\"deletedbyinference\":false},\"collectedfrom\":[{\"dsId\":\"10|openaire____::160a261e5d06fd542c2efcac6e17e08c\",\"dsName\":\"RS Global Journals\"}],\"datasource\":{\"datasourcetype\":{\"code\":\"pubsrepository::journal\",\"label\":\"Journal\"},\"datasourcetypeui\":{\"code\":\"Journal archive\",\"label\":\"Journal archive\"},\"eosctype\":{\"code\":\"Data Source\",\"label\":\"Data Source\"},\"eoscdatasourcetype\":{\"code\":\"Journal archive\",\"label\":\"Journal Archive\"},\"openairecompatibility\":{\"code\":\"hostedBy\",\"label\":\"collected from a compatible aggregator\"},\"officialname\":\"International Journal of Innovative Technologies in Economy\",\"englishname\":\"International Journal of Innovative Technologies in Economy\",\"namespaceprefix\":\"ojs_24128368\",\"latitude\":\"0.0\",\"longitude\":\"0.0\",\"odnumberofitems\":\"0.0\",\"versioning\":false,\"versioncontrol\":false,\"journal\":{\"name\":\"International Journal of Innovative Technologies in Economy\",\"issnPrinted\":\"2412-8368\",\"issnOnline\":\"2414-1305\"},\"thematic\":false,\"contentpolicies\":[{\"code\":\"Journal articles\",\"label\":\"Journal articles\"}],\"consenttermsofuse\":false,\"fulltextdownload\":false}}";
        SolrRecord record = new ObjectMapper().readValue(json, new TypeReference<SolrRecord>() {});

    }

 /*   public static void main(String[] args) throws IOException, CQLParseException, SolrServerException {
        CloudSolrServer solrClient = new CloudSolrServer("beta.solr.openaire.eu:9983");
        solrClient.setDefaultCollection("DMF-index-openaire");

        NamedList<String> queryOpts = new NamedList<String>();

        //q=*:*&start=0&rows=10&cursorMark=*&sort=dateofcollection asc
        queryOpts.add("q", new CqlTranslatorImpl().getTranslatedQuery("objIdentifier = acm_________::0002c24f82c295e925a2bdf7bbf49bfc").asLucene());
        queryOpts.add("start", "0");
        queryOpts.add("rows", "1");
        queryOpts.add("fl", "__result");
        queryOpts.add("shards.tolerant","true");
        queryOpts.add("cursorMark", "*");
        queryOpts.add("sort", "__indexrecordidentifier asc");


        //queryOpts.add("q", new CqlTranslatorImpl().getTranslatedQuery("oaftype exact project").asLucene());
        NamedList<String> extraOpts = new NamedList<String>();

        QueryResponse resp = solrClient.query(SolrParams.toSolrParams(queryOpts));

        System.out.println("results " + resp.getResults().size());

        String cursorMark = "*";
        String nextCursorMark = "";

        int curs = 0;
        while (!cursorMark.equals(nextCursorMark)) {
            System.out.println("cursor " + cursorMark);
            System.out.println("next cursor " + nextCursorMark);
            cursorMark = nextCursorMark;
            for (int i = 0; i < resp.getResults().size(); i++) {
                String result = ((ArrayList<String>) resp.getResults().get(i).get("__result")).get(0);
                //System.out.println(result);
                resp = solrClient.query(SolrParams.toSolrParams(queryOpts));
            }
            nextCursorMark = resp.getNextCursorMark();
            queryOpts.add("cursorMark", nextCursorMark);

            System.out.println("CURS " + curs);
            curs ++;

        }


        //System.out.println((new CqlTranslatorImpl().getTranslatedQuery("objIdentifier = acm_________::0002c24f82c295e925a2bdf7bbf49bfc").asLucene()));



        //extraOpts.add("start", "1");
       // extraOpts.add("rows", "10");
       // extraOpts.addAll(queryOpts);

        //queryOpts.add("facet", "true");
        //TranslatedQuery translatedQuery = new CqlTranslatorImpl().getTranslatedQuery("oaftype=result sortBy resultdateofacceptance/sort.descending");

     //   queryOpts.add("q", "oaftype=project");
        //queryOpts.add("facet", "true");
        //queryOpts.add("facet.mincount", "1");
        //queryOpts.add("fq", "popularity");



//        queryOpts.put("fq", new CqlTranslatorImpl().getTranslatedQuery("").asLucene());
       // queryOpts.add("facet.field", "contextid");
       //  queryOpts.add("facet.field", "contextname");
       //  queryOpts.add("facet.mincount", "1");
       //  queryOpts.add("facet.threads", "10");
       // System.out.println(translatedQuery.getOptions().getSort().getMode());
       // System.out.println(translatedQuery.getOptions().getSort().getField());

        //queryOpts.add("sort", translatedQuery.getOptions().getSort().getField() + " " + translatedQuery.getOptions().getSort().getMode() );



/*        QueryResponse resp = null;
        synchronized (solrClient) {
            resp = solrClient.query(SolrParams.toSolrParams(extraOpts));
        }*/
//        System.out.println("time: " + resp.getElapsedTime());
    //System.out.println("results: " + resp.getResults());

/*      System.out.println(resp.getFacetField("contextname").getValueCount());

        for (FacetField.Count count:resp.getFacetField("contextname").getValues())
            System.out.println(count.getName() + " : " +  count.getCount());


        int max = -12;

        for (FacetField field:resp.getFacetFields()) {
            if (field.getValueCount() > max)
                max = field.getValueCount();

        }

        System.out.println("max: " + max);
*/
    //   }

//    @Override
//    public EPR getEpr() {
//        return epr;
//   }

    private String formatColumn(String value) {
        if (value== null) return "";
        return value;
    }

    private String formatColumn(List<String> value) {
        if (value == null) return "";
        if (value.size()>1) return String.join(";", value);
        return value.get(0);
    }
}

class BrowseField {
    String name;

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getCount() {
        return count;
    }

    public void setCount(String count) {
        this.count = count;
    }

    String id;
    String count;


}