package eu.dnetlib.data.search.utils.vocabulary;

import eu.dnetlib.data.search.transform.config.Configuration;
import org.apache.log4j.Logger;

import java.util.*;

public class VocabularyManagerImpl implements VocabularyManager {
	
	private Configuration config = null;
	
	//key-> vocabulary name, value-> vocabulary loader
	private Map<String, VocabularyLoader> vocabularyLoaderMap = new HashMap<String, VocabularyLoader>();

	private Map<String, String> underCreation = new HashMap<String, String>();
	//key-> vocabulary name, value-> vocabulary item
	//this vocabulary class contains the  fields needed to load the domain.Vocab object
	private Map<String, Vocabulary> vocabularyMap = new HashMap<String, Vocabulary>();
	
	//key-> vocabulary name, value-> vocabulary object per locale
	private Map<String, Map<Locale, eu.dnetlib.domain.enabling.Vocabulary>> pathMap = new HashMap<String, Map<Locale, eu.dnetlib.domain.enabling.Vocabulary>>();
	
	private List<String> indexVocabularies = new ArrayList<String>();
	
	private ISVocabularyLoader isVocabularyLoader;
	private IndexVocabularyLoader indexVocabularyLoader;	
	private LocalVocabularyLoader localVocabularyLoader;
	
	private static Logger logger = Logger.getLogger(VocabularyManagerImpl.class);
	
	public void init() {
		for(String vocabularyName: config.getIsVocabularyMap().keySet()) {
			vocabularyLoaderMap.put(vocabularyName, isVocabularyLoader);
			vocabularyMap.put(vocabularyName, config.getIsVocabularyMap().get(vocabularyName));
		}
		
		for(String vocabularyName: config.getIndexVocabularyMap().keySet()) {
			vocabularyLoaderMap.put(vocabularyName, indexVocabularyLoader);	
			vocabularyMap.put(vocabularyName, config.getIndexVocabularyMap().get(vocabularyName));
			
			indexVocabularies.add(vocabularyName);
		}

		for(String vocabularyName: config.getLocalVocabularyMap().keySet()) {
			vocabularyLoaderMap.put(vocabularyName, localVocabularyLoader);	
			vocabularyMap.put(vocabularyName, config.getLocalVocabularyMap().get(vocabularyName));
		}

	}
	
	public String translate(String name, String l, String encoding, String default_value){
		Locale locale = new Locale(l);
		eu.dnetlib.domain.enabling.Vocabulary vocabulary = getVocabulary(name, locale);

		if(vocabulary == null) {
			return (default_value.length() == 0)? encoding: default_value;
		}
		String term = vocabulary.getEnglishName(encoding);

		if(term != null && !term.trim().isEmpty()) {
			return term;
		}
		
		vocabulary = getVocabulary(name, config.getDefaultLocale());

		if(vocabulary == null) {
			return (default_value.length() == 0)? encoding: default_value;
		}

		term = vocabulary.getEnglishName(encoding);

		if(term != null && !term.trim().isEmpty()){
			return term;
		}

		return (default_value.length() == 0)? encoding: default_value;
	}
	
	public eu.dnetlib.domain.enabling.Vocabulary getVocabulary(String vocabularyName, Locale locale) {
		eu.dnetlib.domain.enabling.Vocabulary myVocabulary = null;
		
		//if vocabulary loader does not exist, return null
		VocabularyLoader vocabularyLoader = vocabularyLoaderMap.get(vocabularyName);
		if (vocabularyLoader == null) {
			return null;
		}
		
		//if locale is not in the configuration file, change it with the default
		if (!config.getLocales().contains(locale)) {
			locale = config.getDefaultLocale();
		}
		
		//if it is an index vocabulary use the default locale only!
		if (indexVocabularies.contains(vocabularyName)) {
			locale = config.getDefaultLocale();
		}
		
		//if vocabulary does not exist, return null
		Vocabulary vocabulary = vocabularyMap.get(vocabularyName);
		if (vocabulary == null) {
			return null;
		}
		
		Map<Locale, eu.dnetlib.domain.enabling.Vocabulary> localePathMap = pathMap.get(vocabularyName);
		
		try {
			if (localePathMap == null) {
				localePathMap = new HashMap<Locale, eu.dnetlib.domain.enabling.Vocabulary>();
			}
			if (underCreation.get(vocabularyName+locale) == null) { //first time to load the vocabulary
				underCreation.put(vocabularyName+locale,"");
				myVocabulary = vocabularyLoader.loadVocabulary(vocabulary, locale);

				localePathMap.put(locale, myVocabulary);
				pathMap.put(vocabularyName, localePathMap);
			
			} else {
				myVocabulary = localePathMap.get(locale);
				if (myVocabulary == null && underCreation.get(vocabularyName + locale) == null) { //first time to load the vocabulary for this locale
					underCreation.put(vocabularyName+locale,"");
					myVocabulary = vocabularyLoader.loadVocabulary(vocabulary, locale);
					localePathMap.put(locale, myVocabulary);
				}
			}
			
		} catch (Exception e) {
			logger.error("Cannot load vocabulary " + vocabularyName + locale, e);

		} finally {
			if (underCreation.get(vocabularyName+locale) != null && !underCreation.get(vocabularyName+locale).isEmpty()) {
				underCreation.put(vocabularyName+locale, null);
			}
		}

		//logger.info("Vocabulary " + vocabularyName + locale + " was created");

		return myVocabulary;
	}
	
	public eu.dnetlib.domain.enabling.Vocabulary getVocabulary(String vocabularyName, Locale locale, boolean loadanyway) {
		eu.dnetlib.domain.enabling.Vocabulary myVocabulary = null;
		
		//if vocabulary loader does not exist, return null
		VocabularyLoader vocabularyLoader = vocabularyLoaderMap.get(vocabularyName);
		if (vocabularyLoader == null) {
			return null;
		}
		
		//if locale is not in the configuration file, change it with the default
		if (!config.getLocales().contains(locale)) {
			locale = config.getDefaultLocale();
		}
		
		//if it is an index vocabulary use the default locale only!
		if (indexVocabularies.contains(vocabularyName)) {
			locale = config.getDefaultLocale();
		}
		
		//if vocabulary does not exist, return null
		Vocabulary vocabulary = vocabularyMap.get(vocabularyName);
        //logger.debug("vocabulary map " + vocabularyMap.keySet());
        //logger.debug("loading vocabulary " + vocabularyName);
        //logger.debug("vocabulary " + vocabulary);

		if (vocabulary == null) {
			return null;
		}
		
		Map<Locale, eu.dnetlib.domain.enabling.Vocabulary> localePathMap = pathMap.get(vocabularyName);
		
		try {
			if (localePathMap == null) { //first time to load the vocabulary
				localePathMap = new HashMap<Locale, eu.dnetlib.domain.enabling.Vocabulary>();				
				myVocabulary = vocabularyLoader.loadVocabulary(vocabulary, locale);
				localePathMap.put(locale, myVocabulary);
				pathMap.put(vocabularyName, localePathMap);
			
			} else {
				myVocabulary = localePathMap.get(locale);
				
				if (myVocabulary == null || loadanyway) { //first time to load the vocabulary for this locale, or load anyway
					myVocabulary = vocabularyLoader.loadVocabulary(vocabulary, locale);
					localePathMap.put(locale, myVocabulary);
				}
			}
			
		} catch (Exception e) {
			logger.error("Cannot load vocabulary " + vocabularyName + " with locale " + locale, e);
		}

		return myVocabulary;
	}

	public void setConfig(Configuration config) {
		this.config = config;
	}

	public void setIsVocabularyLoader(ISVocabularyLoader isVocabularyLoader) {
		this.isVocabularyLoader = isVocabularyLoader;
	}

	public void setIndexVocabularyLoader(IndexVocabularyLoader indexVocabularyLoader) {
		this.indexVocabularyLoader = indexVocabularyLoader;
	}

	public void setLocalVocabularyLoader(LocalVocabularyLoader localVocabularyLoader) {
		this.localVocabularyLoader = localVocabularyLoader;
	}

	public Map<String, Vocabulary> getVocabulariesMap(){
		return this.vocabularyMap;
	}
	
	@Override
	public Set<String> getVocabularyNames() {
		return this.vocabularyMap.keySet();
	}

	@Override
	public List<Locale> getLocales(){
		return this.config.getLocales();
	}
	
}
