package eu.dnetlib.validator2.validation.guideline.openaire;

import eu.dnetlib.validator2.result_models.RequirementLevel;
import eu.dnetlib.validator2.result_models.StandardResult;
import eu.dnetlib.validator2.result_models.ValidationResult;
import eu.dnetlib.validator2.validation.guideline.*;
import eu.dnetlib.validator2.validation.utils.RegexValuePredicate;
import eu.dnetlib.validator2.validation.utils.ResultUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import java.util.*;
import java.util.stream.Collectors;

import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE;
import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE_TO_N;
import static eu.dnetlib.validator2.validation.utils.SupportedRegExs.COMPILED_LICENSE_CONDITION_REG_EX;

public final class FAIR_Data_GuidelinesProfile extends AbstractOpenAireProfile {

    private static final Logger logger = LoggerFactory.getLogger(FAIR_Data_GuidelinesProfile.class);

    private static final String[] rightsURIList = {
            "info:eu-repo/semantics/closedAccess",
            "info:eu-repo/semantics/embargoedAccess",
            "info:eu-repo/semantics/restrictedAccess",
            "info:eu-repo/semantics/openAccess"
    };

    private static final String[] IDENTIFIER_TYPES = {
            "ARK", "arXiv", "bibcode", "DOI", "EAN13", "EISSN", "Handle", "IGSN", "ISBN",
            "ISSN", "ISTC", "LISSN", "ISSNL", "LSID", "PISSN", "PMID", "PURL", "UPC", "URL", "URN", "WOS",
    };

    private static final String[] RELATION_TYPES = {
            "IsCitedBy", "Cites", "IsSupplementTo", "IsSupplementedBy", "IsContinuedBy",
            "Continues", "IsDescribedBy", "Describes", "HasMetadata", "IsMetadataFor", "HasVersion",
            "IsVersionOf", "IsNewVersionOf", "IsPreviousVersionOf", "IsPartOf", "HasPart", "IsReferencedBy",
            "References", "IsDocumentedBy", "Documents", "IsCompiledBy", "Compiles", "IsVariantFormOf",
            "IsOriginalFormOf", "IsIdenticalTo", "IsReviewedBy", "Reviews", "IsDerivedFrom", "IsSourceOf",
            "IsRequiredBy", "Requires"
    };

    private static final String[] RESOURCE_IDENTIFIER_TYPES = {
            "ARK", "DOI", "Handle", "IGSN", "arXiv", "PURL", "URL", "URN", "PMID"
    };

    //    persistent identifier
//    private static final ElementSpec F1_01D_SPEC = Builders.
//            forMandatoryElement("identifier", ONE_TO_N).
////            withMandatoryAttribute("identifierType", new PIDCheckValuePredicate()).
//            withMandatoryAttribute("identifierType", RESOURCE_IDENTIFIER_TYPES).
//            build();

//    persistent identifier
    private static final ElementSpec F1_01D_SPEC = Builders
            .forMandatoryElement("identifier", ONE).inContext("metadata", "oai_datacite", "payload", "resource")
            .withMandatoryAttribute("identifierType", RESOURCE_IDENTIFIER_TYPES)
            .build();

//    // To be implemented: In the case of OpenAIRE we suppose it succeeds
//    private static final ElementSpec F4_01M_SPEC = Builders.
//            forMandatoryElement("identifier", ONE).
//            withMandatoryAttribute("identifierType", RESOURCE_IDENTIFIER_TYPES).
//            build();

    private static final ElementSpec A1_01M_SPEC = Builders.
            forMandatoryElement("rights", ONE_TO_N).
            withMandatoryAttribute("rightsURI", rightsURIList).
            build();

//    // To be implemented
//    private static final ElementSpec A2_01M_SPEC = Builders.
//            forMandatoryElement("subject", ONE_TO_N).
//            withMandatoryAttribute("subjectScheme").
//            withMandatoryAttribute("schemeURI").
//            build();

//    // To be implemented
//    private static final ElementSpec I1_01M_SPEC = Builders.
//            forMandatoryElement("subject", ONE_TO_N).
//            build();

//    // To be implemented
//    // I1_02M Metadata uses semantic resources
//    private static final ElementSpec I1_02M_SPEC = Builders.
//            forMandatoryElement("subject", ONE).
//            build();

    private static final ElementSpec I3_01M_SPEC_1 = Builders.
            forMandatoryElement("relatedIdentifier", ONE_TO_N).
            withMandatoryAttribute("relatedIdentifierType", IDENTIFIER_TYPES).
            withMandatoryAttribute("relationType", RELATION_TYPES).
            build();

//    private static final ElementSpec I3_01M_SPEC_2 = Builders.
//            forMandatoryElement("creator", ONE_TO_N).
//            withMandatoryAttribute("creatorName").
//            withMandatoryAttribute("nameIdentifier", NAME_IDENTIFIER_SCHEMES).
//            withMandatoryAttribute("nameIdentifierScheme", NAME_IDENTIFIER_SCHEMES).
//            withMandatoryAttribute("schemeURI", NAME_IDENTIFIER_SCHEMES_URIS).
//            withMandatoryAttribute("affiliation", NAME_IDENTIFIER_SCHEMES).
//            build();

    // TO BE IMPLEMENTED

    private static final ElementSpec R1_1_01M_SPEC = Builders.
            forMandatoryElement("rights", ONE_TO_N)
            .atPosition(ElementPosition.FIRST)
            .withMandatoryAttribute("rightsURI")
            .allowedValues(new RegexValuePredicate(COMPILED_LICENSE_CONDITION_REG_EX))
            .build();

    //TODO: weights for guidelines haven't been finalized. They've been given an arbitrary value of 10.
    public static SyntheticGuideline F1_01D = SyntheticGuideline.of("(meta)data are assigned a globally unique and persistent identifier", "F1", "description", "https://w3id.org/fair/principles/latest/F1", "F",
            10, RequirementLevel.MANDATORY, F1_01D_SPEC);

    public static SyntheticGuideline A1_01M = SyntheticGuideline.of("(meta)data are retrievable by their identifier using a standardized communications protocol", "A1", "description", "https://w3id.org/fair/principles/latest/A1", "A",
            10, RequirementLevel.MANDATORY, F1_01D_SPEC);

    public static SyntheticGuideline A1_01_1M = SyntheticGuideline.of("the protocol is open, free, and universally implementable", "A1.1", "description", "https://w3id.org/fair/principles/latest/A1.1", "A",
            10, RequirementLevel.MANDATORY, F1_01D_SPEC);

    //    public static SyntheticGuideline A1_01_2M = SyntheticGuideline.of("the protocol allows for an authentication and authorization procedure, where necessary", "A1.2", "description", "https://w3id.org/fair/principles/latest/A1.2", "A",
    //            10, RequirementLevel.MANDATORY, A1_01_2M_SPEC);

    //    public static SyntheticGuideline A2_01M = SyntheticGuideline.of("metadata are accessible, even when the data are no longer available", "A2", "description", "https://w3id.org/fair/principles/latest/A2", "A",
    //            10, RequirementLevel.MANDATORY, A2_01M_SPEC);


    public static SyntheticGuideline I3_01M_1 = SyntheticGuideline.of("(meta)data include qualified references to other (meta)data", "I3", "description", "https://w3id.org/fair/principles/latest/I3", "I",
        10, RequirementLevel.MANDATORY, I3_01M_SPEC_1);

    //    public static SyntheticGuideline I3_01M_2 = SyntheticGuideline.of("I3_01M_2", "description", "https://guidelines.openaire.eu",
//                                                                5, RequirementLevel.MANDATORY, I3_01M_SPEC_2);

//    public static SyntheticGuideline R1_01M = SyntheticGuideline.of("R1_01M", "description", "https://guidelines.openaire.eu",
//                                                                10, RequirementLevel.MANDATORY, R1_01M_SPEC);

    public static SyntheticGuideline R1_1_01M = SyntheticGuideline.of("(meta)data are released with a clear and accessible data usage license", "R1.1", "description", "https://w3id.org/fair/principles/latest/R1.1", "R",
            10, RequirementLevel.MANDATORY, R1_1_01M_SPEC);

    private static final List<Guideline<Document>> GUIDELINES = Collections.unmodifiableList(
            Arrays.asList(
                    F1_01D,
                    new F2_01M_SPEC(),
                    new F3_01M_SPEC(),
                    new F4M_SPEC(),
                    A1_01M,
                    A1_01_1M,
                    new I1M_SPEC(),
                    new I2_01M_SPEC(),
                    I3_01M_1,
                    new R1_01M_SPEC(),
                    R1_1_01M,
                    new R1_2_01M_SPEC(),
                    new R1_3_01M_SPEC()
            )
    );

    private static final Map<String, Guideline> GUIDELINE_MAP = GUIDELINES.stream().
            collect(Collectors.toMap(Guideline::getName, (guideline) -> guideline));

    private static final int MAX_SCORE = GUIDELINES.stream().map(Guideline::getWeight).reduce(0, Integer::sum);

    public FAIR_Data_GuidelinesProfile() {
        super("OpenAIRE FAIR Guidelines for Data Repositories Profile");
    }

    @Override
    public Collection<? extends Guideline<Document>> guidelines() {
        return GUIDELINES;
    }

    /**
     *
     * @param guidelineName
     * @return
     */
    @Override
    public Guideline guideline(String guidelineName) {
        return GUIDELINE_MAP.get(guidelineName);
    }

    @Override
    public int maxScore() {
        return MAX_SCORE;
    }
}


//// TODO this goes to FAIRProfile
/*class MetadataCompleteness extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(MetadataCompleteness.class);


    public MetadataCompleteness() {
        super("MetadataCompleteness", 40);
    }

    @Override
    public StandardResult validate(String id, Document t) {
        DataArchiveGuidelinesV2Profile profile = new DataArchiveGuidelinesV2Profile();

        // <></>ODO: iterate over results and build one StandardResult
        try {
//            System.out.println("Processing MetadataCompleteness...");
            ValidationResult res = profile.validate(id, t);
            Map<String, Result> results = res.results();
            int MaxScoreMetadataCompleteness = (int) ((res.score()*getWeight())/100);

            logger.debug("Max score DataValidator(%): " + res.score());
            logger.debug("Weight FAIRG: " + getWeight());
            logger.debug("Max score MetadataCompleteness: " + MaxScoreMetadataCompleteness);
            logger.debug("\n\n\n\n");

            if ( logger.isDebugEnabled() ) {
                for ( Map.Entry entry : results.entrySet() ) {
                    logger.debug(entry.getKey() + " = " + entry.getValue());
                }
            }

            logger.debug(results.entrySet().stream().
                    map(entry -> entry.getValue() + ": " + entry.getKey()).collect(Collectors.joining("\n")));

            return getResult(MaxScoreMetadataCompleteness);

        } catch (Exception e) {
            System.out.println(e.getMessage());
            System.out.println(e);
            e.printStackTrace();
        }

        return null;
    }

    private static StandardResult getResult(int score) {
        String aa;
        aa = (score > 0) ? "SUCCESS" : "FAILURE";
        return new Result() {
            @Override
            public int score() {
                return score;
            }

            @Override
            public Status status() {
//                    return null;
                return Status.valueOf(aa);
            }

            @Override
            public Iterable<String> warnings() { return null; }

            @Override
            public Iterable<String> errors() {
                return null;
            }

            @Override
            public String internalError() {
                return null;
            }
        };
    }
}*/



class F2_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F2_01M_SPEC.class);

    public F2_01M_SPEC() {
        super("data are described with rich metadata", "F2", "description", "https://w3id.org/fair/principles/latest/F2", "F", 2*5, RequirementLevel.MANDATORY);
    }

    @Override
    public StandardResult validate(String id, Document t) {
        F2_01M profile = new F2_01M();

        // TODO: iterate over results and build one StandardResult
        try {
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();

//            Get actual score and not (%) to incorporate to FAIR score
            final int MaxScoreF2_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            logger.debug("Max score DataValidator(%): " + res_F.getScore());
            logger.debug("Weight FAIRG: " + getWeight());
            logger.debug("Max score F2_01M_SPEC: " + MaxScoreF2_01M_SPEC);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF2_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }


    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class F3_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F3_01M_SPEC.class);


    public F3_01M_SPEC() { super("metadata clearly and explicitly include the identifier of the data it describes", "F3", "description", "https://w3id.org/fair/principles/latest/F3", "F", 10, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        F3_01M profile = new F3_01M();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nMetadata includes the identifier for the data");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreF3_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

//            This is to validate that if one of the checks in the F3_01M class is successful, then the whole FAIR rule becomes successful.
            int MaxScoreF3_01M_SPEC;
            if ((int) res_F.getScore() == 50 ) {
                MaxScoreF3_01M_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            }
            else {
                MaxScoreF3_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);
            }

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF3_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());

        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }
    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {

        return ResultUtils.getNewResult(data, score);
    }
}

class F4M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(F4M_SPEC.class);

    public F4M_SPEC() { super("(meta)data are registered or indexed in a searchable resource", "F4", "description", "https://w3id.org/fair/principles/latest/F4", "F", 10, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        DataArchiveGuidelinesV2Profile profile = new DataArchiveGuidelinesV2Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes the identifier for the data");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreF4M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

//            This is to validate that if one of the checks in the F3_01M class is successfull then the whole FAIR rule becomes successfull
            int MaxScoreF4M_SPEC;
            if ((int) res_F.getScore() == 50 ) {
                MaxScoreF4M_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            }
            else {
                MaxScoreF4M_SPEC = (int) ((res_F.getScore() * getWeight())/100);
            }

            final StandardResult ress = getResult(results.entrySet(), MaxScoreF4M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }
    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {

        return ResultUtils.getNewResult(data, score);
    }
}

class I1M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(I1M_SPEC.class);

    public I1M_SPEC() { super("(meta)data use a formal, accessible, shared, and broadly applicable language for knowledge representation", "I1", "description", "https://w3id.org/fair/principles/latest/I1", "F", 10, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        DataArchiveGuidelinesV2Profile profile = new DataArchiveGuidelinesV2Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes the identifier for the data");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreI1M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

//            This is to validate that if one of the checks in the F3_01M class is successfull then the whole FAIR rule becomes successfull
            int MaxScoreI1M_SPEC;
            if ((int) res_F.getScore() == 50 ) {
                MaxScoreI1M_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            }
            else {
                MaxScoreI1M_SPEC = (int) ((res_F.getScore() * getWeight())/100);
            }

            final StandardResult ress = getResult(results.entrySet(), MaxScoreI1M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }
    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class I2_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(I2_01M_SPEC.class);

    public I2_01M_SPEC() { super("(meta)data use vocabularies that follow FAIR principles", "I2", "description", "https://w3id.org/fair/principles/latest/I2", "I", 5*2, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        I2_01M profile = new I2_01M();

        // TODO: iterate over results and build one StandardResult
        try {
            logger.debug("Metadata uses FAIR-compliant vocabularies");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
//            int MaxScoreI2_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

//            This is to validate that if one of the checks in the F3_01M class is successfull then the whole FAIR rule becomes successfull
            int MaxScoreI2_01M_SPEC;
            if ((int) res_F.getScore() == 50 )
                MaxScoreI2_01M_SPEC = (int) (((2*res_F.getScore()) * getWeight())/100);
            else
                MaxScoreI2_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreI2_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_01M_SPEC.class);


    public R1_01M_SPEC() { super("meta(data) are richly described with a plurality of accurate and relevant attributes", "R1", "description", "https://w3id.org/fair/principles/latest/R1", "R", 3*4, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        R1_01M profile = new R1_01M();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nPlurality of accurate and relevant attributes are provided to allow reuse");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), -1);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_2_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_2_01M_SPEC.class);


    public R1_2_01M_SPEC() { super("(meta)data are associated with detailed provenance", "R1.2", "description", "https://w3id.org/fair/principles/latest/R1.2", "R", 2*5, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        R1_2_01M profile = new R1_2_01M();

        // TODO: iterate over results and build one StandardResult
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_2_01M_SPEC = (int) ((res_F.getScore() * getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), -1);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}

class R1_3_01M_SPEC extends AbstractGuideline<Document> {

    private static final Logger logger = LoggerFactory.getLogger(R1_3_01M_SPEC.class);


    public R1_3_01M_SPEC() { super("(meta)data meet domain-relevant community standards", "R1.3", "description", "https://w3id.org/fair/principles/latest/R1.3", "R", 10, RequirementLevel.MANDATORY); }

    @Override
    public StandardResult validate(String id, Document t) {
        DataArchiveGuidelinesV2Profile profile = new DataArchiveGuidelinesV2Profile();

        // TODO: iterate over results and build one Guideline.Result
        try {
//            System.out.println("\nMetadata includes provenance information according to a cross-community language");
            ValidationResult res_F = profile.validate(id, t);
            Map<String, StandardResult> results = res_F.getResults();
            int MaxScoreR1_3_01M_SPEC = (int) ((res_F.getScore()*getWeight())/100);

            final StandardResult ress = getResult(results.entrySet(), MaxScoreR1_3_01M_SPEC);

            return new StandardResult(ress.getScore(), ress.getStatus(), ress.getWarnings(), ress.getErrors(), ress.getInternalError());
        } catch (Exception e) {
            logger.error("", e);
            return null;
        }
    }

    private static StandardResult getResult(Set<Map.Entry<String, StandardResult>> data, int score) {
        return ResultUtils.getNewResult(data, score);
    }
}
