package eu.dnetlib.validator2.validation.guideline.openaire;

import eu.dnetlib.validator2.validation.guideline.*;
import eu.dnetlib.validator2.validation.utils.MediaTypesValuePredicate;
import org.w3c.dom.Document;

import java.util.*;
import java.util.stream.Collectors;

import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE;
import static eu.dnetlib.validator2.validation.guideline.Cardinality.ONE_TO_N;
import static eu.dnetlib.validator2.validation.utils.SupportedRegExs.COMPILED_BCP47_LANG_TAGS_REG_EX;

public final class R1_01M_LIT_v3 extends AbstractOpenAireProfile {

    private static final String[] publicationTypes = {
            "info:eu-repo/semantics/article",
            "info:eu-repo/semantics/bachelorThesis",
            "info:eu-repo/semantics/masterThesis",
            "info:eu-repo/semantics/doctoralThesis",
            "info:eu-repo/semantics/book",
            "info:eu-repo/semantics/bookPart",
            "info:eu-repo/semantics/review",
            "info:eu-repo/semantics/conferenceObject",
            "info:eu-repo/semantics/lecture",
            "info:eu-repo/semantics/workingPaper",
            "info:eu-repo/semantics/preprint",
            "info:eu-repo/semantics/report",
            "info:eu-repo/semantics/annotation",
            "info:eu-repo/semantics/contributionToPeriodical",
            "info:eu-repo/semantics/patent",
            "info:eu-repo/semantics/other"
    };

    private static final String[] publicationVersions = {
            "info:eu-repo/semantics/draft",
            "info:eu-repo/semantics/submittedVersion",
            "info:eu-repo/semantics/acceptedVersion",
            "info:eu-repo/semantics/publishedVersion",
            "info:eu-repo/semantics/updateVersion"
    };

    private static final ElementSpec RESOURCE_TYPE_SPEC = Builders.
            forMandatoryElement("dc:type", ONE).
            atPosition(ElementPosition.FIRST).
            allowedValues(publicationTypes).
            build();

    private static final ElementSpec RESOURCE_VERSION_SPEC = Builders.
            forMandatoryElement("dc:type", ONE).
            allowedValues(publicationVersions).
            build();

    private static final ElementSpec FORMAT_SPEC = Builders.
            forMandatoryElement("dc:format", ONE_TO_N).
            allowedValues(new MediaTypesValuePredicate()).
            build();

//    Method for data Not applicable in Literature
    private static final ElementSpec DESCRIPTION_SPEC = Builders. // With very low weight
            forMandatoryElement("dc:description", ONE_TO_N).
            build();

    //TODO: weights for guidelines haven't been finalized. They've been given an arbitrary value of 1.
    public static SyntheticGuideline R1_01M_LIT_1 = SyntheticGuideline.of("ResourceType", 4, RESOURCE_TYPE_SPEC);
    public static SyntheticGuideline R1_01M_LIT_2 = SyntheticGuideline.of("Format", 2, RESOURCE_VERSION_SPEC);
    public static SyntheticGuideline R1_01M_LIT_3 = SyntheticGuideline.of("Description", 2, FORMAT_SPEC);
    public static SyntheticGuideline R1_01M_LIT_4 = SyntheticGuideline.of("Description", 2, DESCRIPTION_SPEC);

    private static final List<Guideline<Document>> GUIDELINES = Collections.unmodifiableList(
            Arrays.asList(
                    R1_01M_LIT_1,
                    R1_01M_LIT_2,
                    R1_01M_LIT_3,
                    R1_01M_LIT_4
            )
    );

    private static final Map<String, Guideline> GUIDELINE_MAP = GUIDELINES.
            stream().
            collect(Collectors.toMap(Guideline::getName, (guideline) -> guideline));

    private static final int MAX_SCORE = GUIDELINES.stream().map(Guideline::getWeight).reduce(0, Integer::sum);

    public R1_01M_LIT_v3() {
        super("Plurality of accurate and relevant attributes are provided to allow reuse");
    }

    @Override
    public Collection<? extends Guideline<Document>> guidelines() {
        return GUIDELINES;
    }

    /**
     *
     * @param guidelineName
     * @return
     */
    @Override
    public Guideline guideline(String guidelineName) {
        return GUIDELINE_MAP.get(guidelineName);
    }

    @Override
    public int maxScore() {
        return MAX_SCORE;
    }
}